{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module AST (
	Pat, Pat' (NamePat, AndPat, NotPat, SpecificPat, VoidPat),
	Pattern, Pattern' (Pattern),
	Op, Op' (ListTerm, VariableTerm, IntegerTerm, StringTerm, LambdaTerm,
	    BindTerm, SeqTerm, ApplicationTerm, ProductTerm, VoidTerm,
	    AbstractTerm, PlanTerm, AnnotatedTerm,
	    GenericTypeTerm),
	Term, Term'(TM),
	Copat, Copat' (NameCopat, OrCopat, NotCopat, PackageCopat, UnreachedCopat),
	Copattern, Copattern' (Copattern),
	Coop, Coop' (ColistTerm, CovariableTerm, ColambdaTerm, CobindTerm, CoseqTerm,
	    CoapplicationTerm, SumTerm, UnreachedTerm,
	    CoabstractTerm, UseTerm, CoannotatedTerm,
	    AbstractTypeTerm),
	Coterm, Coterm'(TO),
	TypeExpression, TypeExpression'(TypeExpression),
	XExpr(XExpr),

	listSubexpressions, listSubcoexpressions,
	Expression, boundNamesOf,

	reproduce) where

import List
import Maybe
import Control.Monad.State

import Basic
import Position
import Type
import Unify
import Kind

data Pat' anno
	    = NamePat VarName
	    | AndPat [Maybe (Pattern' anno)]
	    | VoidPat
	    | NotPat (Copattern' anno)
	    | SpecificPat (TypeExpression' anno) (Pattern' anno)
type Pat = Pat' (Type, Span)
data Pattern' anno = Pattern anno (Pat' anno)
type Pattern = Pattern' (Type, Span)
data Op' anno
	    = ListTerm [Term' anno]
            | VariableTerm VarName
            | IntegerTerm Integer
	    | StringTerm String
            | LambdaTerm (Pattern' anno) {- param -} (Term' anno)
	    | BindTerm (Term' anno) (Term' anno) -- :: m s -> (s -> m t) -> m t
	    | SeqTerm (Term' anno) (Term' anno) -- :: m s -> m t -> m t
	    | ApplicationTerm (Term' anno) {- arg -} (Term' anno) {- function -}
	    | VoidTerm
	    | ProductTerm [Term' anno]
	    | AbstractTerm (Copattern' anno) (Term' anno) (Coterm' anno)
	    | PlanTerm (Coterm' anno)
	    | AnnotatedTerm Bool (Term' anno)
	    | GenericTypeTerm anno VarName (Term' anno) -- this has universal type
type Op = Op' (Type, Span)
data Term' anno = TM anno (Op' anno)
type Term = Term' (Type, Span)

data Copat' anno
	    = NameCopat VarName
	    | OrCopat [Maybe (Copattern' anno)]
	    | UnreachedCopat
	    | NotCopat (Pattern' anno)
	    | PackageCopat (TypeExpression' anno) (Copattern' anno)
type Copat = Copat' (Type, Span)
data Copattern' anno = Copattern anno (Copat' anno)
type Copattern = Copattern' (Type, Span)
data Coop' anno
	    = ColistTerm [Coterm' anno]
            | CovariableTerm VarName
            | ColambdaTerm (Copattern' anno) {- param -} (Coterm' anno)
	    | CobindTerm (Coterm' anno) (Coterm' anno) -- :: m s -> (s -> m t) -> m t
	    | CoseqTerm (Coterm' anno) (Coterm' anno) -- :: m s -> m t -> m t
	    | CoapplicationTerm (Coterm' anno) {- arg -} (Coterm' anno) {- cofunction -}
	    | UnreachedTerm
	    | SumTerm [Coterm' anno]
	    | CoabstractTerm (Pattern' anno) (Term' anno) (Coterm' anno)
	    | UseTerm (Term' anno)
	    | CoannotatedTerm Bool (Coterm' anno)
	    | AbstractTypeTerm anno VarName (Coterm' anno) -- this has existential type
type Coop = Coop' (Type, Span)
data Coterm' anno = TO anno (Coop' anno)
type Coterm = Coterm' (Type, Span)

data TypeExpression' anno = TypeExpression anno
type TypeExpression = TypeExpression' (Type, Span)

instance (TShow (exprt (String, Span)), Functor exprt) => TShow (exprt (Type, Span))
    where
	tshow ds dt dk expr = tshow ds dt dk $ fmap decorate expr
	    where
		decorate (tp, spn) = (kshow default_type_limit ds dk tp, spn)
instance (TShow (exprt (String, Span)), Functor exprt) => TShow (exprt Type)
    where
	tshow ds dt dk expr = tshow ds dt dk $ fmap decorate expr
	    where
		decorate tp = (kshow default_type_limit ds dk tp, no_span)
instance (TShow (exprt (String, Span)), Functor exprt, FunctorM exprt, MonadIO m) => MonadicTShow m (exprt (UTRef, Span))
    where
	do_tshow ds dt dk expr = do
		decorated <- liftIO $ fmapM decorate expr
		return (tshow ds dt dk decorated)
	    where
		decorate (tp, spn) = do
			tp_str <- do_kshow default_type_limit ds dk tp
			return (tp_str, spn)
instance TShow (Pat' (String, Span))
    where
	tshow ds dt dk pat = case pat of
		NamePat nm -> nm
		VoidPat -> ""
		AndPat mpats -> concatMap (('&':) . (maybe "_" (\pat -> "(" ++ tshow ds dt dk pat ++ ")"))) mpats ++ "&"
		NotPat copat -> "([" ++ tshow ds dt dk copat ++ "])"
		SpecificPat typ pat -> "@" ++ stp ++ tshow ds dt dk pat
		    where
			stp = tshow ds dt dk typ ++ ":>"
instance TShow (Pattern' (String, Span))
    where
	tshow ds dt dk (Pattern (tp, spn) pat) = bracket $ tshow ds dt dk pat ++ show_t ++ show_spn
	    where
		bracket x = if show_t /= "" || show_spn /= "" then  "(" ++ x ++ ")" else x
		show_spn = case ds of
				NoShowSpan -> ""
				_ | is_span spn -> ", " ++ show spn
				otherwise -> ""
		show_t = case tp of
				"" -> ""
				shown_tp | dt -> ":" ++ shown_tp
				otherwise -> ""
instance TShow (Term' (String, Span))
    where
	tshow ds dt dk (TM (typ, spn) op) = bracket $ tshow (nest ds) dt dk op ++ show_t ++ show_spn
	    where
		show_spn = case ds of
			NoShowSpan -> ""
			_ | is_span spn -> ", " ++ show spn
			otherwise -> ""
		show_t = case typ of
			    "" -> ""
			    shown_typ | dt || anno -> ":" ++ shown_typ
			    otherwise -> ""
		anno = case op of
			    AnnotatedTerm _ _ -> True
			    otherwise -> False
		bracket x = if anno then "(" ++ x ++ ")" else x
instance TShow (Op' (String, Span)) where
	tshow ds dt dk x = case x of
		ListTerm ms -> "(" ++ concat (map (\m -> "," ++ tshow ds dt dk m) ms) ++ ",)"
		IntegerTerm n -> show n
		StringTerm str -> "\"" ++ str ++ "\""
		VariableTerm str -> str
		LambdaTerm pat m -> "\\" ++ tshow ds dt dk pat ++ "->" ++ bracket (tshow ds dt dk m)
		BindTerm def (TM ltp (LambdaTerm pat m)) -> "{" ++ tshow ds dt dk def ++ "=:" ++ tshow ds dt dk pat ++ ";" ++ tshow ds dt dk m ++ (if dt then show_t ltp else "") ++ "}"
		BindTerm def term -> "{" ++ tshow ds dt dk def ++ ">>=" ++ tshow ds dt dk term ++ "}"
		SeqTerm def term -> "{" ++ tshow ds dt dk def ++ ";" ++ tshow ds dt dk term ++ "}"
		ApplicationTerm v f -> tshow ds dt dk v ++ " " ++ tshow ds dt dk f
		PlanTerm c -> "([" ++ tshow ds dt dk c ++ "])"
		VoidTerm -> "()"
		ProductTerm ms -> "(" ++ concat (map (\m -> "&" ++ tshow ds dt dk m) ms) ++ "&)"
		AbstractTerm pat t c -> "(" ++ tshow ds dt dk pat ++ "<+" ++ tshow ds dt dk t ++ "$" ++ tshow ds dt dk c ++ ")"
		AnnotatedTerm _ t -> tshow ds dt dk t
		GenericTypeTerm (kt,_) nm t -> "@" ++ nm ++ ":>" ++ tshow ds dt dk t ++ (if dk then (dropWhile (/= ':') kt) else "")
	    where
		bracket x = if dt then "(" ++ x ++ ")" else x
		show_t (typ,_) = case typ of
			    "" -> ""
			    shown_typ -> ":" ++ shown_typ

instance TShow (Copat' (String, Span))
    where
	tshow ds dt dk pat = case pat of
		NameCopat nm -> nm
		UnreachedCopat -> ""
		OrCopat mpats -> concatMap (('|':) . (maybe "_" (\pat -> "[" ++ tshow ds dt dk pat ++ "]"))) mpats ++ "|"
		NotCopat copat -> "[(" ++ tshow ds dt dk copat ++ ")]"
		PackageCopat typ pat -> "[@" ++ stp ++ tshow ds dt dk pat ++ "]"
		    where
			stp = tshow ds dt dk typ ++ "<:"
instance TShow (Copattern' (String, Span))
    where
	tshow ds dt dk (Copattern (tp, spn) copat) = bracket $ tshow ds dt dk copat ++ show_t ++ show_spn
	    where
		bracket x = if show_t /= "" || show_spn /= "" then  "[" ++ x ++ "]" else x
		show_spn = case ds of
				NoShowSpan -> ""
				_ | is_span spn -> ", " ++ show spn
				otherwise -> ""
		show_t = case tp of
				"" -> ""
				shown_tp | dt -> ":" ++ shown_tp
				otherwise -> ""
instance TShow (Coterm' (String, Span))
    where
	tshow ds dt dk (TO (typ, spn) coop) = bracket $ tshow (nest ds) dt dk coop ++ show_t ++ show_spn
	    where
		show_spn = case ds of
			NoShowSpan -> ""
			_ | is_span spn -> ", " ++ show spn
			otherwise -> ""
		show_t = case typ of
			    "" -> ""
			    shown_typ | dt || anno -> ":" ++ shown_typ
			    otherwise -> ""
		anno = case coop of
			    CoannotatedTerm _ _ -> True
			    otherwise -> False
		bracket x = if anno then "[" ++ x ++ "]" else x

instance TShow (Coop' (String, Span)) where
	tshow ds dt dk x = case x of
		ColistTerm ms -> "[" ++ concat (map (\m -> "," ++ tshow ds dt dk m) ms) ++ ",]"
		CovariableTerm str -> str
		ColambdaTerm copat m -> "\\" ++ tshow ds dt dk copat ++ "<+" ++ tshow ds dt dk m
		CobindTerm def (TO ltp (ColambdaTerm copat m)) -> "{" ++ tshow ds dt dk def ++ "=:" ++ tshow ds dt dk copat ++ ";" ++ tshow ds dt dk m ++ "}"
		CobindTerm def term -> "{" ++ tshow ds dt dk def ++ ">>=" ++ tshow ds dt dk term ++ "}"
		CoseqTerm def term -> "{" ++ tshow ds dt dk def ++ ";" ++ tshow ds dt dk term ++ "}"
		CoapplicationTerm v f -> tshow ds dt dk v ++ " " ++ tshow ds dt dk f
		UseTerm t -> "[(" ++ tshow ds dt dk t ++ ")]"
		UnreachedTerm -> "[]"
		SumTerm ts -> "[" ++ concat (map (\t -> "|" ++ tshow ds dt dk t) ts) ++ "|]"
		CoabstractTerm copat t c -> "[" ++ tshow ds dt dk copat ++ "->" ++ tshow ds dt dk t ++ "$" ++ tshow ds dt dk c ++ "]"
		CoannotatedTerm _ t -> tshow ds dt dk t
		AbstractTypeTerm (kt,_) nm t -> "@" ++ nm ++ "<:" ++ tshow ds dt dk t ++ (if dk then (dropWhile (/= ':') kt) else "")

instance TShow (TypeExpression' (String, Span)) where
	tshow ds dt dk (TypeExpression (typ, _)) = typ

instance Parameterized Pat where
    namesOf pat = case pat of
	NamePat nm -> []
	VoidPat -> []
	AndPat mpats -> foldr union [] (map (maybe [] namesOf) mpats)
	NotPat copat -> namesOf copat
	SpecificPat tp pat -> namesOf tp `union` namesOf pat
    usedOf pat = case pat of
	NamePat nm -> [nm]
	VoidPat -> []
	AndPat mpats -> foldr union [] (map (maybe [] usedOf) mpats)
	NotPat copat -> usedOf copat
	SpecificPat tp pat -> usedOf tp `union` usedOf pat
instance Parameterized Pattern where
    namesOf pattern@(Pattern (tp,sp) pat) = (namesOf tp \\ boundNamesOf pattern) `union` namesOf pat
    usedOf (Pattern (tp,sp) pat) = usedOf tp `union` usedOf pat

instance Parameterized Op
  where
    namesOf op = case op of
	    ListTerm ts -> foldr union [] (map namesOf ts)
	    VariableTerm str -> [str]
	    IntegerTerm _ -> []
	    StringTerm str -> []
	    LambdaTerm pat t -> (namesOf t `union` namesOf pat) \\ (boundNamesOf pat)
	    BindTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    SeqTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    ApplicationTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    VoidTerm -> []
	    ProductTerm ts -> foldr union [] (map namesOf ts)
	    PlanTerm c -> namesOf c
	    AbstractTerm copat t c -> (namesOf t `union` namesOf c `union` namesOf copat) \\ boundNamesOf copat
	    AnnotatedTerm _ t -> namesOf t
	    GenericTypeTerm k nm t -> nm `delete` namesOf t
    usedOf op = case op of
	    ListTerm ts -> foldr union [] (map usedOf ts)
	    VariableTerm str -> [str]
	    IntegerTerm _ -> []
	    StringTerm str -> []
	    LambdaTerm pat t -> usedOf pat `union` usedOf t
	    BindTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    SeqTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    ApplicationTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    VoidTerm -> []
	    ProductTerm ts -> foldr union [] (map usedOf ts)
	    PlanTerm c -> usedOf c
	    AbstractTerm copat t c -> usedOf copat `union` usedOf t `union` usedOf c
	    AnnotatedTerm _ t -> usedOf t
	    GenericTypeTerm k nm t -> [nm] `union` usedOf t

instance Parameterized Term
  where
    namesOf (TM (typ,sp) op) = namesOf typ `union` namesOf op
    usedOf (TM (typ,sp) op) = usedOf typ `union` usedOf op

instance Parameterized Copat where
    namesOf copat = case copat of
	NameCopat nm -> []
	UnreachedCopat -> []
	OrCopat mcopats -> foldr union [] (map (maybe [] namesOf) mcopats)
	NotCopat pat -> namesOf pat
	PackageCopat tp copat -> namesOf tp `union` namesOf copat
    usedOf copat = case copat of
	NameCopat nm -> [nm]
	UnreachedCopat -> []
	OrCopat mcopats -> foldr union [] (map (maybe [] usedOf) mcopats)
	NotCopat pat -> usedOf pat
	PackageCopat tp copat -> usedOf tp `union` usedOf copat
instance Parameterized Copattern where
    namesOf copattern@(Copattern (tp,sp) copat) = (namesOf tp \\ boundNamesOf copattern) `union` namesOf copat
    usedOf (Copattern (tp,sp) copat) = usedOf tp `union` usedOf copat

instance Parameterized Coop
  where
    namesOf coop = case coop of
	    ColistTerm ts -> foldr union [] (map namesOf ts)
	    CovariableTerm str -> [str]
	    ColambdaTerm copat c -> namesOf c \\ boundNamesOf copat
	    CobindTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    CoseqTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    CoapplicationTerm term1 term2 -> union (namesOf term1) (namesOf term2)
	    UnreachedTerm -> []
	    SumTerm cs -> foldr union [] (map namesOf cs)
	    UseTerm t -> namesOf t
	    CoabstractTerm pat t c -> (namesOf t `union` namesOf c) \\ boundNamesOf pat
	    CoannotatedTerm _ term -> namesOf term
	    AbstractTypeTerm k nm t -> nm `delete` namesOf t
    usedOf coop = case coop of
	    ColistTerm ts -> foldr union [] (map usedOf ts)
	    CovariableTerm str -> [str]
	    ColambdaTerm copat t -> usedOf copat `union` usedOf t
	    CobindTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    CoseqTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    CoapplicationTerm term1 term2 -> union (usedOf term1) (usedOf term2)
	    UnreachedTerm -> []
	    SumTerm cs -> foldr union [] (map usedOf cs)
	    UseTerm t -> usedOf t
	    CoabstractTerm pat t c -> usedOf pat `union` usedOf t `union` usedOf c
	    CoannotatedTerm _ t -> usedOf t
	    AbstractTypeTerm k nm t -> [nm] `union` usedOf t

instance Parameterized Coterm
  where
    namesOf (TO (typ,sp) coop) = namesOf typ `union` namesOf coop
    usedOf (TO (typ,sp) coop) = usedOf typ `union` usedOf coop

instance Parameterized TypeExpression
  where
    namesOf (TypeExpression (typ,sp)) = namesOf typ
    usedOf (TypeExpression (typ,sp)) = usedOf typ

class (Parameterized e, Substitutable (Either Term Coterm) e) => Expression e where
  transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> e -> e
  foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> e -> x
  typeOf :: e -> Maybe Type

emap :: Expression e => (forall t . (Expression t) => (t -> t)) -> e -> e
emap fn = fn . transformSubexpressions (emap fn)

instance Expression Op
  where
    transformSubexpressions f op = case op of
	    ListTerm terms -> ListTerm (map f terms)
	    LambdaTerm pat term -> LambdaTerm (f pat) (f term)
	    BindTerm term1 term2 -> BindTerm (f term1) (f term2)
	    SeqTerm term1 term2 -> SeqTerm (f term1) (f term2)
	    ApplicationTerm arg func -> ApplicationTerm (f arg) (f func)
	    ProductTerm ts -> ProductTerm (map f ts)
	    PlanTerm ctm -> PlanTerm (f ctm)
	    AbstractTerm copat t c -> AbstractTerm (f copat) (f t) (f c)
	    AnnotatedTerm preu term -> AnnotatedTerm preu (f term)
	    GenericTypeTerm k nm t -> GenericTypeTerm k nm (f t)
	    otherwise -> op
    foldExpression f op = f op folded
	where
	    folded = case op of
		ListTerm terms -> map (foldExpression f) terms
		LambdaTerm pat term -> [foldExpression f pat, foldExpression f term]
		BindTerm term1 term2 -> [foldExpression f term1, foldExpression f term2]
		SeqTerm term1 term2 -> [foldExpression f term1, foldExpression f term2]
		ApplicationTerm arg func -> [foldExpression f arg, foldExpression f func]
		ProductTerm ts -> map (foldExpression f) ts
		PlanTerm ctm -> [foldExpression f ctm]
		AbstractTerm copat t c -> [foldExpression f copat, foldExpression f t, foldExpression f c]
		AnnotatedTerm _ term -> [foldExpression f term]
		GenericTypeTerm k nm t -> [foldExpression f t]
		otherwise -> []
    typeOf op = Nothing

instance Substitutable (Either Term Coterm) Op where
    substitute name sub op = case op of
	    LambdaTerm pat lt	| name `elem` boundNamesOf pat -> op
				| otherwise -> dflt
	    AbstractTerm copat st sc
				| name `elem` boundNamesOf copat -> op
				| otherwise -> dflt
	    otherwise -> dflt
        where
	    dflt = transformSubexpressions (substitute name sub) op

instance Expression Term
  where
    transformSubexpressions f (TM tp op) = TM tp $ f op
    foldExpression f trm@(TM tp op) = f trm [foldExpression f op]
    typeOf (TM (tp,sp) tm) = Just tp
instance Substitutable (Either Term Coterm) Term where
    substitute name sub (TM (tp, span) op) = case op of
	    VariableTerm nm
		| nm == name -> case sub of
			Left subee -> (subee :: Term)
			Right subee -> assertWay ByValue ByName (show span) undefined
	    otherwise -> TM (tp, span) $ substitute name sub op

instance Expression Coop
  where
    transformSubexpressions f coop = case coop of
	    ColistTerm ts -> ColistTerm (map f ts)
	    CovariableTerm str -> coop
	    ColambdaTerm copat lt -> ColambdaTerm (f copat) (f lt)
	    CobindTerm term1 term2 -> CobindTerm (f term1) (f term2)
	    CoseqTerm term1 term2 -> CoseqTerm (f term1) (f term2)
	    CoapplicationTerm term1 term2 -> CoapplicationTerm (f term1) (f term2)
	    UseTerm term -> UseTerm (f term)
	    UnreachedTerm -> UnreachedTerm
	    SumTerm cs -> SumTerm (map f cs)
	    CoabstractTerm pat t c -> CoabstractTerm (f pat) (f t) (f c)
	    CoannotatedTerm preu t -> CoannotatedTerm preu (f t)
	    AbstractTypeTerm k nm t -> AbstractTypeTerm k nm (f t)
    foldExpression f coop = f coop folded
	where
	    folded = case coop of
		ColistTerm ts -> map (foldExpression f) ts
		CovariableTerm str -> []
		ColambdaTerm copat lt -> [foldExpression f copat, foldExpression f lt]
		CobindTerm term1 term2 -> [foldExpression f term1, foldExpression f term2]
		CoseqTerm term1 term2 -> [foldExpression f term1, foldExpression f term2]
		CoapplicationTerm term1 term2 -> [foldExpression f term1, foldExpression f term2]
		UseTerm term -> [foldExpression f term]
		UnreachedTerm -> []
		SumTerm cs -> map (foldExpression f) cs
		CoabstractTerm pat t c -> [foldExpression f pat, foldExpression f t, foldExpression f c]
		CoannotatedTerm _ t -> [foldExpression f t]
		AbstractTypeTerm k nm t -> [foldExpression f t]
    typeOf coop = Nothing
instance Substitutable (Either Term Coterm) Coop where
    substitute name sub coop = case coop of
	    ColambdaTerm copat lt
				| name `elem` boundNamesOf copat -> coop
				| otherwise -> dflt
	    CoabstractTerm pat st sc
				| name `elem` boundNamesOf pat -> coop
				| otherwise -> dflt
	    otherwise -> dflt
        where
	    dflt = transformSubexpressions (substitute name sub) coop

instance Expression Coterm
  where
    transformSubexpressions f (TO tp coop) = TO tp $ f coop
    foldExpression f c@(TO tp coop) = f c [foldExpression f coop]
    typeOf (TO (tp,sp) coop) = Just tp
instance Substitutable (Either Term Coterm) Coterm where
    substitute name sub (TO (tp, span) coop) = case coop of
	    CovariableTerm nm
		| nm == name -> case sub of
			Right subee -> subee
			Left subee -> assertWay ByName ByValue (show span) undefined
	    otherwise -> TO (tp, span) $ substitute name sub coop

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --

class Anyop anno exprt where
    listSubexpressions :: exprt anno -> [Term' anno]
    listSubcoexpressions :: exprt anno -> [Coterm' anno]

class Structured t where
    boundNamesOf :: t -> [VarName]

instance Structured Pattern where
    boundNamesOf (Pattern tp pat) = case pat of
	NamePat nm -> [nm]
	VoidPat -> []
	AndPat mpats -> concatMap (maybe [] boundNamesOf) mpats
	NotPat copat -> boundNamesOf copat
	SpecificPat tp pat -> boundNamesOf pat

instance FunctorM Pat' where
    fmapM f pat = case pat of
	AndPat mpats -> do 
		mpats'<- mapM combine mpats
		return $ AndPat mpats'
	    where
		combine elem = maybe (return Nothing )
					    threadAPat
					    elem
		    where
			threadAPat pat = do
				pat' <- fmapM f pat
				return $ Just pat'
	NotPat copat -> do
		    copat' <- fmapM f copat
		    return $ NotPat copat'
	SpecificPat tp pat -> do
		tp' <- fmapM f tp
		pat' <- fmapM f pat
		return $ SpecificPat tp' pat'
	NamePat nm -> return $ NamePat nm
	VoidPat -> return $ VoidPat
instance Substitutable (Either Term Coterm) Pat where
    substitute name sub p = p
instance Expression Pat
    where
	-- transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> Pat -> Pat
	transformSubexpressions f pat = case pat of
	    AndPat mpats -> AndPat (map (fmap f) mpats)
	    NotPat copat -> NotPat (f copat)
	    SpecificPat tp pat -> SpecificPat (f tp) (f pat)
	    otherwise -> pat
	-- foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> Pat -> x
	foldExpression f pat = f pat foldeds
	    where
		foldeds = case pat of
		    AndPat mpats -> map (foldExpression f) (concatMap maybeToList mpats)
		    NotPat copat -> [foldExpression f copat]
		    SpecificPat tp pat -> [foldExpression f tp, foldExpression f pat]
		    otherwise -> []
	-- typeOf :: Pat -> Maybe Type
	typeOf pat = Nothing
instance FunctorM Pattern' where
    fmapM f (Pattern anno pat) = do
				anno' <- f anno
				pat' <- fmapM f pat
				return $ Pattern anno' pat'
instance Substitutable (Either Term Coterm) Pattern where
    substitute name sub p = p
instance Expression Pattern
    where
	-- transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> Pattern -> Pattern
	transformSubexpressions f (Pattern tp pat) = Pattern tp (f pat)
	-- foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> Pattern -> x
	foldExpression f pattern@(Pattern tp pat) = f pattern [foldExpression f pat]
	-- typeOf :: Pattern -> Maybe Type
	typeOf (Pattern (tp,sp) pat) = Just tp
			
instance FunctorM Op' where
    fmapM f op = case op of
	IntegerTerm n -> return $ IntegerTerm n
	StringTerm str -> return $ StringTerm str
	VariableTerm str -> return $ VariableTerm str
	ListTerm ts -> do
			ts' <- mapM (fmapM f) ts
			return $ ListTerm ts'
	LambdaTerm pat t -> do
			pat' <- fmapM f pat
			t' <- fmapM f t
			return $ LambdaTerm pat' t'
	BindTerm l r -> do
			l' <- fmapM f l
			r' <- fmapM f r
			return $ BindTerm l' r'
	SeqTerm l r -> do
			l' <- fmapM f l
			r' <- fmapM f r
			return $ SeqTerm l' r'
	ApplicationTerm arg func -> do
			arg' <- fmapM f arg
			func' <- fmapM f func
			return $ ApplicationTerm arg' func'
	VoidTerm -> return $ VoidTerm
	ProductTerm ts -> do
			ts' <- mapM (fmapM f) ts
			return $ ProductTerm ts'
	PlanTerm c -> do
			c' <- fmapM f c
			return $ PlanTerm c'
	AbstractTerm pat t c -> do
			pat' <- fmapM f pat
			t' <- fmapM f t
			c' <- fmapM f c
			return $ AbstractTerm pat' t' c'
	AnnotatedTerm preu t -> do
			t' <- fmapM f t
			return $ AnnotatedTerm preu t'
	GenericTypeTerm k nm t -> do
			t' <- fmapM f t
			k' <- f k
			return $ GenericTypeTerm k' nm t'

instance FunctorM Term' where
    fmapM f (TM anno tm) = do
			-- error "HI"
			anno' <- f anno
			tm' <- fmapM f tm
			return $ TM anno' tm'

instance Structured Copattern where
    boundNamesOf (Copattern (typ,sp) copat) = case copat of
	NameCopat nm -> [nm]
	UnreachedCopat -> []
	OrCopat mcopats -> concatMap (maybe [] boundNamesOf) mcopats
	NotCopat pat -> boundNamesOf pat
	PackageCopat tp copat -> boundNamesOf copat
instance FunctorM Copattern' where
    fmapM f (Copattern anno copat) = do
				anno' <- f anno
				copat' <- fmapM f copat
				return $ Copattern anno' copat'
instance FunctorM Copat' where
    fmapM f copat = case copat of
	OrCopat mcopats -> do
		mcopats' <- mapM combine mcopats
		return $ OrCopat mcopats'
	    where
		combine elem = maybe (return Nothing)
						threadACopat
						elem
		    where
			threadACopat copat = do
				copat' <- fmapM f copat
				return $ Just copat'
	NotCopat pat -> do
		    pat' <- fmapM f pat
		    return $ NotCopat pat'
	PackageCopat tp copat -> do
		tp' <- fmapM f tp
		copat' <- fmapM f copat
		return $ PackageCopat tp' copat'
	NameCopat nm -> return $ NameCopat nm
	UnreachedCopat -> return $ UnreachedCopat
	
instance Substitutable (Either Term Coterm) Copat where
    substitute name sub p = p
instance Expression Copat
    where
	-- transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> Copat -> Copat
	transformSubexpressions f copat = case copat of
	    OrCopat mcopats -> OrCopat (map (fmap f) mcopats)
	    NotCopat pat -> NotCopat (f pat)
	    PackageCopat tp copat -> PackageCopat (f tp) (f copat)
	    otherwise -> copat
	-- foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> Copat -> x
	foldExpression f copat = f copat foldeds
	    where
		foldeds = case copat of
		    OrCopat mcopats -> map (foldExpression f) (concatMap maybeToList mcopats)
		    NotCopat pat -> [foldExpression f pat]
		    PackageCopat tp copat -> [foldExpression f tp, foldExpression f copat]
		    otherwise -> []
	-- typeOf :: Copat -> Maybe Type
	typeOf copat = Nothing
instance Substitutable (Either Term Coterm) Copattern where
    substitute name sub p = p
instance Expression Copattern
    where
	-- transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> Copattern -> Copattern
	transformSubexpressions f (Copattern tp copat) = Copattern tp (f copat)
	-- foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> Copattern -> x
	foldExpression f copattern@(Copattern tp copat) = f copattern [foldExpression f copat]
	-- typeOf :: Copattern -> Maybe Type
	typeOf (Copattern (tp,sp) copat) = Just tp
instance Substitutable (Either Term Coterm) TypeExpression where
    substitute name sub t = t
instance Expression TypeExpression
    where
	-- transformSubexpressions :: (forall t . (Expression t) => (t -> t)) -> TypeExpression -> TypeExpression
	transformSubexpressions f (TypeExpression tp) = TypeExpression tp
	-- foldExpression :: (forall t. (Expression t) => (t -> [x] -> x)) -> TypeExpression -> x
	foldExpression f t@(TypeExpression tp) = f t []
	-- typeOf :: TypeExpression -> Maybe Type
	typeOf (TypeExpression (tp,sp)) = Just tp

instance FunctorM Coop' where
    fmapM f coop = case coop of
	CovariableTerm str -> return $ CovariableTerm str
	ColistTerm ts -> do
			ts_ <- mapM (fmapM f) ts
			return $ ColistTerm ts_
	ColambdaTerm copat t -> do
			copat' <- fmapM f copat
			t' <- fmapM f t
			return $ ColambdaTerm copat' t'
	CobindTerm l r -> do
			l' <- fmapM f l
			r' <- fmapM f r
			return $ CobindTerm l' r'
	CoseqTerm l r -> do
			l' <- fmapM f l
			r' <- fmapM f r
			return $ CoseqTerm l' r'
	CoapplicationTerm arg func -> do
			arg' <- fmapM f arg
			func' <- fmapM f func
			return $ CoapplicationTerm arg' func'
	UseTerm tm -> do
			tm' <- fmapM f tm
			return $ UseTerm tm'
	UnreachedTerm -> return UnreachedTerm
	SumTerm cs -> do
			cs' <- mapM (fmapM f) cs
			return $ SumTerm cs'
	CoabstractTerm pat t c -> do
			pat' <- fmapM f pat
			t' <- fmapM f t
			c' <- fmapM f c
			return $ CoabstractTerm pat' t' c'
	CoannotatedTerm preu ctm -> do
			ctm' <- fmapM f ctm
			return $ CoannotatedTerm preu ctm'
	AbstractTypeTerm k nm t -> do
			t' <- fmapM f t
			k' <- f k
			return $ AbstractTypeTerm k' nm t'
instance FunctorM Coterm' where
    fmapM f (TO anno tm) = do
			anno' <- f anno
			tm' <- fmapM f tm
			return $ TO anno' tm'
instance FunctorM TypeExpression' where
    fmapM f (TypeExpression anno) = do
				anno' <- f anno
				return $ TypeExpression anno'

instance Anyop anno Op' where
    -- listSubexpressions :: Op -> [Term]
    listSubexpressions op = case op of
			    ListTerm terms -> terms
			    LambdaTerm pat term -> [term]
			    BindTerm t1 t2 -> [t1, t2]
			    SeqTerm t1 t2 -> [t1, t2]
			    ApplicationTerm arg func -> [arg, func]
			    ProductTerm ts -> ts
			    AbstractTerm copat t c -> [t]
			    AnnotatedTerm _ t -> [t]
			    GenericTypeTerm k nm t -> [t]
			    otherwise -> []
    listSubcoexpressions op = case op of
			    PlanTerm ct -> [ct]
			    AbstractTerm copat t c -> [c]
			    otherwise -> []

instance Anyop anno Coop' where
    -- listSubcoexpressions :: Coop -> [Coterm]
    listSubcoexpressions op = case op of
			    ColistTerm terms -> terms
			    ColambdaTerm copat term -> [term]
			    CobindTerm t1 t2 -> [t1, t2]
			    CoseqTerm t1 t2 -> [t1, t2]
			    CoapplicationTerm arg func -> [arg, func]
			    SumTerm cs -> cs
			    CoabstractTerm pat t c -> [c]
			    CoannotatedTerm _ term -> [term]
			    AbstractTypeTerm k nm c -> [c]
			    otherwise -> []
    listSubexpressions coop = case coop of
			    UseTerm tm -> [tm]
			    CoabstractTerm pat t c -> [t]
			    otherwise -> []

class HasType t where
    type_of :: t -> Type
instance HasType Type where
    type_of tp = tp
instance (HasType tp) => HasType (tp, Span) where
    type_of (t,_) = type_of t
instance (HasType tp) => HasType (TypeExpression' tp) where
    type_of (TypeExpression x) = type_of x

instance Positionable Term where
	set_pos pos (TM (t,_) x) = TM (t,pos) x
	get_span (TM (t,pos) x) = pos
instance Positionable Coterm where
	set_pos pos (TO (t,_) x) = TO (t,pos) x
	get_span (TO (t,pos) x) = pos
instance Positionable Pattern where
	set_pos pos (Pattern (t,_) x) = Pattern (t,pos) x
	get_span (Pattern (t,pos) x) = pos
instance Positionable Copattern where
	set_pos pos (Copattern (t,_) x) = Copattern (t,pos) x
	get_span (Copattern (t,pos) x) = pos
instance Positionable TypeExpression where
	set_pos pos (TypeExpression (t,_)) = TypeExpression (t,pos)
	get_span (TypeExpression (t,pos)) = pos

class Reproducible t where
    reproduce :: t -> String

instance Reproducible Term where
    reproduce x@(TM (tp,sp) op) = case op of
	ListTerm ms -> revise x (map SomePos ms)
	LambdaTerm pat m -> revise x [SomePos pat, SomePos m]
	BindTerm def (TM ltp (LambdaTerm pat m))
	    -> revise x [SomePos def, SomePos pat, SomePos m]
	BindTerm def term -> revise x [SomePos def, SomePos term]
	SeqTerm def term -> revise x [SomePos def, SomePos term]
	ApplicationTerm v f -> revise x [SomePos v, SomePos f]
	PlanTerm c -> revise x [SomePos c]
	ProductTerm ms -> revise x (map SomePos ms)
	AbstractTerm pat t c -> reproduce pat ++ "<+" ++ reproduce t ++ "$" ++ reproduce c
	AnnotatedTerm _ t -> case tp of
		Type k sp NoT -> revise x [SomePos t]
		Type k sp _ -> revise x [SomePos t, SomePos tp]
	GenericTypeTerm k nm t -> revise x [SomePos t]
	otherwise -> revise x []
instance Reproducible Coterm where
    reproduce x@(TO (tp,sp) coop) = case coop of
	ColistTerm ms -> revise x (map SomePos ms)
	ColambdaTerm pat t -> "\\" ++ reproduce pat ++ "<+" ++ reproduce t
	CobindTerm def (TO ltp (ColambdaTerm copat m))
	    -> revise x [SomePos def, SomePos copat, SomePos m]
	CobindTerm def term -> revise x [SomePos def, SomePos term]
	CoseqTerm def term -> revise x [SomePos def, SomePos term]
	CoapplicationTerm func arg -> ini ++ reproduce arg ++ " " ++ reproduce func ++ fin
	    where
		Span _ _ ini = pos_span (start_pos_of x) (start_pos_of func)
		Span _ _ fin = pos_span (end_pos_of arg) (end_pos_of x)
	UseTerm t -> revise x [SomePos t]
	SumTerm ts -> revise x (map SomePos ts)
	CoabstractTerm copat t c -> revise x [SomePos copat, SomePos t, SomePos c]
	CoannotatedTerm _ t -> case tp of
		Type k sp NoT -> revise x [SomePos t]
		Type k sp _ -> revise x [SomePos t, SomePos tp]
	AbstractTypeTerm k nm t -> "@" ++ nm ++ ":>" ++ reproduce t
	otherwise -> revise x []
instance Reproducible Pattern where
    reproduce x@(Pattern tp pat) = case pat of
	SpecificPat (TypeExpression (Type NoKind _ NoT, _)) pat -> revise x [SomePos pat]
	SpecificPat typ pat -> revise x [SomePos typ, SomePos pat]
	otherwise -> revise x []
instance Reproducible Copattern where
    reproduce x@(Copattern tp pat) = case pat of
	PackageCopat (TypeExpression (Type NoKind _ NoT, _)) pat -> "@" ++ reproduce pat
	PackageCopat tp pat -> "@" ++ reproduce tp ++ ":>" ++ reproduce pat
	otherwise -> revise x []
instance Reproducible TypeExpression where
    reproduce x@(TypeExpression (tp,sp)) = revise x [SomePos tp]
instance Reproducible Type where
    reproduce x@(Type k sp tp) = case tp of
	AndT _ ts -> revise x (map SomePos ts)
	OrT ts -> revise x (map SomePos ts)
	FunctionT arg result -> revise x [SomePos arg, SomePos result]
	CofunctionT arg result -> reproduce arg ++ "<+" ++ reproduce result
	ApplicationT constructor arg -> revise x [SomePos constructor, SomePos arg]
	LambdaT kind nm typ -> revise x [SomePos typ]
	YNT tp -> revise x [SomePos tp]
	YVT tp -> revise x [SomePos tp]
	ExistsT kind nm tp -> revise x [SomePos tp]
	ForallT kind nm tp -> revise x [SomePos tp]
	otherwise -> revise x []

data SomePositionable = forall t. (Positionable t, Reproducible t) => SomePos t

revise :: (Positionable a) => a -> [SomePositionable] -> String
revise x ss = revise' start end ss
    where
	Span start end str = get_span x
	revise' start end terms = case terms of
		[] -> let Span _ _ str = pos_span start end in str
		(SomePos t):ts -> up_to_t ++ revt ++ revise' start' end ts
		    where
			revt = reproduce t
			Span _ _ up_to_t = pos_span start (start_pos_of t)
			start' = end_pos_of t

instance Functor Pat' where
    fmap xfrm p = case p of
	NamePat nm -> NamePat nm
	AndPat mps -> AndPat (map (fmap (fmap xfrm)) mps)
	VoidPat -> VoidPat
	NotPat copat -> NotPat (fmap xfrm copat)
	SpecificPat tp pat -> SpecificPat (fmap xfrm tp) (fmap xfrm pat)
instance Functor Pattern' where
    fmap xfrm (Pattern anno pat) = Pattern (xfrm anno) (fmap xfrm pat)
instance Functor Op' where
    fmap xfrm op = case op of
	ListTerm ts -> ListTerm (map (fmap xfrm) ts)
	VariableTerm nm -> VariableTerm nm
	IntegerTerm n -> IntegerTerm n
	StringTerm str -> StringTerm str
	LambdaTerm pat tm -> LambdaTerm (fmap xfrm pat) (fmap xfrm tm)
	BindTerm tm1 tm2 -> BindTerm (fmap xfrm tm1) (fmap xfrm tm2)
	SeqTerm tm1 tm2 -> SeqTerm (fmap xfrm tm1) (fmap xfrm tm2)
	ApplicationTerm arg func -> ApplicationTerm (fmap xfrm arg) (fmap xfrm func)
	VoidTerm -> VoidTerm
	ProductTerm ts -> ProductTerm (map (fmap xfrm) ts)
	AbstractTerm copat tm cotm -> AbstractTerm (fmap xfrm copat) (fmap xfrm tm) (fmap xfrm cotm)
	PlanTerm cotm -> PlanTerm (fmap xfrm cotm)
	AnnotatedTerm preu tm -> AnnotatedTerm preu (fmap xfrm tm)
	GenericTypeTerm k nm tm -> GenericTypeTerm (xfrm k) nm (fmap xfrm tm)
instance Functor Term' where
    fmap xfrm (TM anno op) = TM (xfrm anno) (fmap xfrm op)
instance Functor Copat' where
    fmap xfrm copat = case copat of
	NameCopat nm -> NameCopat nm
	OrCopat mcopats -> OrCopat (map (fmap (fmap xfrm)) mcopats)
	UnreachedCopat -> UnreachedCopat
	NotCopat pat -> NotCopat (fmap xfrm pat)
	PackageCopat tp copat -> PackageCopat (fmap xfrm tp) (fmap xfrm copat)
instance Functor Copattern' where
   fmap xfrm (Copattern anno copat) = Copattern (xfrm anno) (fmap xfrm copat)
instance Functor Coop' where
   fmap xfrm coop = case coop of
	ColistTerm cotms -> ColistTerm (map (fmap xfrm) cotms)
	CovariableTerm nm -> CovariableTerm nm
	ColambdaTerm param_pat cotm -> ColambdaTerm (fmap xfrm param_pat) (fmap xfrm cotm)
	CobindTerm cotm1 cotm2 -> CobindTerm (fmap xfrm cotm1) (fmap xfrm cotm2)
	CoseqTerm cotm1 cotm2 -> CoseqTerm (fmap xfrm cotm1) (fmap xfrm cotm2)
	CoapplicationTerm arg func -> CoapplicationTerm (fmap xfrm arg) (fmap xfrm func)
	UnreachedTerm -> UnreachedTerm
	SumTerm cotms -> SumTerm (map (fmap xfrm) cotms)
	CoabstractTerm pat tm cotm -> CoabstractTerm (fmap xfrm pat) (fmap xfrm tm) (fmap xfrm cotm)
	UseTerm tm -> UseTerm (fmap xfrm tm)
	CoannotatedTerm preu cotm -> CoannotatedTerm preu (fmap xfrm cotm)
	AbstractTypeTerm k nm cotm -> AbstractTypeTerm (xfrm k) nm (fmap xfrm cotm)
instance Functor Coterm' where
    fmap xfrm (TO anno coop) = TO (xfrm anno) (fmap xfrm coop)
instance Functor TypeExpression' where
    fmap xfrm (TypeExpression tp) = TypeExpression (xfrm tp)

{-
   XExpr was invented because unify needs a FunctorM type, but Term and Coterm
   were not cutting it.

   ERROR "./Main.hs":82 - Type error in application
   *** Expression     : unify tu kc tk uc ulimit native_builtin_types ev'
   *** Term           : ev'
   *** Type           : Term' (Type,Span)
   *** Does not match : a Type
-}
data XExpr exprt t = XExpr (exprt (t, Span))
instance (FunctorM exprt) => FunctorM (XExpr exprt) where
	-- fmapM :: (Monad m) => (a -> m b) -> XExpr exprt a -> m (XExpr exprt b)
	fmapM f (XExpr trm) = do
				trm' <- fmapM (\(trm,spn) -> do {trm' <- f trm; return (trm',spn)}) trm
				return (XExpr trm')
instance (MonadicTShow m (exprt (UTRef, Span)), FunctorM exprt, Functor exprt, MonadIO m, Functor m, TShow (exprt (String, Span))) => MonadicTShow m (XExpr exprt UTRef) where
	do_tshow ds dt dk (XExpr trm) = do_tshow ds dt dk trm
