{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Kind (
	KindParameterized,
	KindParameterizedAugmented,
	kindMatchesOf,
	kinds_match,
	augmentedKindMatchesOf,
	unify_kinds, unify_kind_matches, unify_kind_match,
	apply_kind_unifier, Unifier(Unifier),
	more_kvars, relink,
	prep_kind, extract_kind, post_prep_kind
	)
    where
import List
import Data.Map
import Control.Monad.State
import Data.IORef
import IO
import Maybe

import Basic
import Type
import Position

instance Functor Knd where
    fmap f k = case k of
	NoKind -> NoKind
	Proposition -> Proposition
	Predicate k1 k2 -> Predicate (f k1) (f k2)
	VarKind ids k -> VarKind ids k
instance FunctorM Knd where
    fmapM f k = case k of
	NoKind -> return NoKind
	Proposition -> return Proposition
	Predicate k1 k2 -> do
		k1' <- f k1
		k2' <- f k2
		return $ Predicate k1' k2'
	VarKind ids k -> return $ VarKind ids k

instance Show Kind where
    show (Kind k) = show k
instance Show (Knd Kind) where
    show k = show (fmap show k :: Knd String)
instance Show (Knd String) where
    show k = case k of
	    NoKind -> "_"
	    Proposition -> "*"
	    Predicate k1 k2 -> "(" ++ k1 ++ "->" ++ k2 ++ ")"
	    VarKind id ref -> show id

readIOKRef :: (MonadIO m) => UTRef -> m (Knd UTRef)
readIOKRef ref = do
    UType k sp tp <- liftIO $ readIORef ref
    return k
writeIOKRef :: (MonadIO m) => UTRef -> Knd UTRef -> m ()
writeIOKRef ref kval = do
    UType _ sp tp <- liftIO $ readIORef ref
    liftIO $ writeIORef ref (UType kval sp tp)
    cc "3" ref

instance (MonadIO m) => MonadicShow m UTRef where
    do_show kref = do
	k <- readIOKRef kref
	do_show k
instance (MonadIO m) => MonadicShow m (Knd UTRef) where
    do_show k = do
		k' <- fmapM do_show k
		return (show k')

prep_kind :: Kind -> StateT [TypeId] IO UTRef
prep_kind (Kind knd) = do
		knd' <- fmapM prep_kind knd
		ref <- liftIO $ newIORef (UType knd' no_span NoT)
		post_prep_kind ref knd'
		return ref				

post_prep_kind :: UTRef -> Knd UTRef -> StateT [TypeId] IO ()
post_prep_kind ref knd = do
		case knd of
			NoKind -> do
				mv <- liftState get
				case mv of
				    (v:_) -> do
					liftState (modify tail)
					(UType _ sp tp) <- liftIO $ readIORef ref
					liftIO $ writeIORef ref (UType (VarKind v ref) sp tp)
					cc "2" ref
				    otherwise -> return ()
			VarKind ids kref -> do -- kref points to a VarKind from the original clique.
			    cc "0015" kref
			    kvars <- fmap (kref :) (more_kvars kref kref)
			    if not (ref `elem` kvars) then
				    Kind.relink (last kvars) ref
				else return ()
			    cc "15" ref
			    cc "015" kref
			_ -> return ()

extract_kind :: (MonadIO m) => UTRef -> m Kind
extract_kind ref = do
		(UType k sp tp) <- liftIO $ readIORef ref
		k' <- fmapM extract_kind k
		return (Kind k')

class KindParameterized t where
	-- The String is to identify where the match arose.
	kindMatchesOf :: Map VarName UTRef -> t -> StateT [TypeId] IO [(UTRef, UTRef, String)]

class KindParameterizedAugmented t where
	augmentedKindMatchesOf :: Maybe Handle -> Bool -> Integer
		-> MonadModel -> UseIOType -> UseFunctionTypes
		-> Map VarName (Way, TypeInContext) -> Map VarName UTRef
		-> t -> StateT [TypeId] IO [(UTRef, UTRef, String)]

data UnifierResult k = GoodUnifier (Unifier k) | BadUnifier Kind Kind
data Unifier k = Unifier k k String
instance Show (Unifier k) where
    show (Unifier var kind display) = display

unify_kinds verbose_types verbose limit mm uio ufn needed builtins stmt = do
	-- kind_matches <- liftIO $ kindMatchesOf builtins stmt
	-- matches = {- trace ("getting kindMatchesOf") $ -} augmentedKindMatchesOf builtins empty e
	kind_matches <- augmentedKindMatchesOf verbose_types (isJust verbose) limit mm uio ufn builtins empty stmt
	progress <- unify_kind_matches verbose kind_matches
	return stmt

unify_kind_matches :: Maybe Handle -> [(UTRef, UTRef, String)] -> StateT [TypeId] IO Bool
unify_kind_matches verbose matches = do
    case matches of
	(match:matches') -> do
	    aside verbose $ "kind unification processing a match"
	    progress <- unify_kind_match verbose match
	    when progress $ aside verbose $ "unification progress"
	    progress' <- unify_kind_matches verbose matches'
	    when progress $ aside verbose $ "unification progress'"
	    return (progress || progress')
	[] -> return False

unify_kind_match :: Maybe Handle -> (UTRef, UTRef, String) -> StateT [TypeId] IO Bool
unify_kind_match verbose (tref1, tref2, wher) = do
	mus <- unify2 tref1 tref2
	case mus of
	    Nothing -> do
		k1 <- readIOKRef tref1
		k2 <- readIOKRef tref2
		ts1 <- do_show tref1
		ts2 <- do_show tref2
		die  ("can't unify " ++ ts1 ++ " with " ++ ts2 ++ "\nat " ++ wher)
	    Just us -> do
		progresses <- mapM (apply_kind_unifier verbose) us
		return (or progresses)

-- Returns a boolean indicating progress.
apply_kind_unifier :: Maybe Handle -> Unifier UTRef -> StateT [TypeId] IO Bool
apply_kind_unifier verbose (Unifier kvar_ref kind_ref wher) = do
	    kvar <- readIOKRef kvar_ref
	    kval <- readIOKRef kind_ref
	    case verbose of
		Just h -> do
			kvar_str <- do_show kvar_ref
			str_for_kind_ref <- do_show kind_ref
			liftIO $ hPutStrLn h ("Applying unifier to variable " ++ kvar_str ++ ": " ++ str_for_kind_ref)
		Nothing -> return ()
	    case kvar of
		VarKind id1 kvar' -> do
		    kvars <- fmap (kvar_ref :) (more_kvars kvar_ref kvar_ref)
		    case kval of
			VarKind id2 kvar' -> do
			    kvars' <- fmap (kvar' :) (more_kvars kvar' kvar')
			    if (kvar_ref `elem` kvars' || kvar' `elem` kvars) then return () -- die "variable unified with itself"
				else do
				    -- Join the circular lists.
				    relink (last kvars) kvar'
				    relink (last kvars') kvar_ref
				    -- Update the list of ids.
				    id1' <- get_id kvar_ref
				    id2' <- get_id kvar'
				    mapM_ (set_id (min id1' id2')) (kvars ++ kvars')
				    when (id1' /= id2') $ aside verbose $ "Subsumed kind variable " ++ show (max id1' id2')
									++ " into " ++ show (min id1' id2') ++ "."
			    return True
			otherwise -> do
			    mapM_ (\kvar -> writeIOKRef kvar kval) kvars

			    -- A variable has been updated; unify some more.
			    unify_kind_match verbose
				(kvar_ref, kind_ref, "generated in apply_kind_unifier")
		otherwise -> do -- A variable has been updated; unify some more.
		    unify_kind_match verbose (kvar_ref, kind_ref, "generated in apply_kind_unifier")
	where
	    get_id kvar = do
		VarKind id next_kvar <- readIOKRef kvar
		return id
	    set_id id kvar = do
		VarKind _ next_kvar <- readIOKRef kvar
		writeIOKRef kvar (VarKind id next_kvar)

{-
more_kvars :: (MonadIO m) => UTRef -> UTRef -> m [k]
more_kvars to_kvar from_kvar = do
    this_var <- readIOKRef from_kvar
    case this_var of
	VarKind ids kvarref -> if kvarref == to_kvar then return []
			    else do
				more <- more_kvars to_kvar kvarref
				return (kvarref : more)
	otherwise -> die "unified a non-type-variable"
-}

-- Return a list of all refs in the list up to, but not including, the to_kvar.
more_kvars :: (Functor m, MonadIO m) => {- String -> -} UTRef -> UTRef -> m [UTRef]
more_kvars {- msg -} to_kvar from_kvar = safe_more_kvars True from_kvar to_kvar from_kvar
    where
	msg = "0"
	safe_more_kvars pause stopper to_kvar from_kvar = do
	    this_var <- readIOKRef from_kvar
	    case this_var of
		VarKind ids kvarref -> if kvarref == to_kvar then return []
				    else do
					stopper' <- if pause then return stopper
					    else do
						    VarKind ids next <- readIOKRef stopper
						    if kvarref == next then do
								die $ "internal error: entry into cycle at more_kvars " ++ msg
							else return ()
						    return next
					more <- safe_more_kvars (not pause) stopper' to_kvar kvarref
					return (kvarref : more)
		otherwise -> die $ "unified a non-kind-variable in more_kvars " ++ msg

relink :: (MonadIO m) => UTRef -> UTRef -> m ()
relink from_var to = do
    VarKind ids kvar <- readIOKRef from_var
    writeIOKRef from_var (VarKind ids to)

-- Nothing: inconsistent
-- []: already unified
-- [(x,t):xs]: Transformation from VarT x to t is implied.

-- The context is a list of type name pairs taken from quantifiers
-- for universal and existential types that we have delved into.
-- If the lists are different, they must be accounted for when
-- unifying with named, or potentially named types.
-- In each pair, the left member corresponds to t1 and the right to t2.

unify2 :: UTRef -> UTRef -> StateT [TypeId] IO (Maybe [Unifier UTRef])
unify2 k1 k2 = u2 k1 k2
  where
    -- u2 :: Kind -> Kind -> Maybe [Unifier k]
    u2 kref1 kref2 = do
	{- trace (strictList_ $ "unifying kinds: "
	    ++ show kind1 ++ " with " ++ show kind2 ++ "\n") $ -}
      kind1 <- readIOKRef kref1
      kind2 <- readIOKRef kref2
      case (kind1,kind2) of
	(NoKind, _) -> return $ Just []
	(_, NoKind) -> return $ Just []
	(VarKind ids1 n1, VarKind ids2 n2) | n1 == n2 -> return $ Just []
	(VarKind ids1 n1, _) -> do
					ks2 <- do_show kref2
					return $ Just [Unifier n1 kref2 ("(" ++ show ids1 ++ "," ++ ks2 ++ ")")]
	(_, VarKind ids2 n2) -> do
					ks1 <- do_show kref1
					return $ Just [Unifier n2 kref1 ("(" ++ show ids2 ++ "," ++ ks1 ++ ")")]
	(Proposition, Proposition) -> return $ Just []
	(Predicate from1 to1, Predicate from2 to2) -> do
				maybe_x1 <- u2 from1 from2
				maybe_x2 <- u2 to1 to2
				return $ do
					    x1 <- maybe_x1
					    x2 <- maybe_x2
					    return $ x1 ++ x2
	_ -> return Nothing

kinds_match :: UTRef -> UTRef -> StateT [TypeId] IO Bool
kinds_match kref1 kref2 = do
	k1 <- readIOKRef kref1
	k2 <- readIOKRef kref2
	case (k1,k2) of
	    (Proposition, Proposition) -> return True
	    (Predicate k11 k12, Predicate k21 k22) -> do
		m1 <- kinds_match k11 k21
		m2 <- kinds_match k12 k22
		return $ m1 && m2
	    (VarKind ids1 _, VarKind ids2 _) -> return $ ids1 == ids2
	    _ -> return False
