{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Strict (strict) where
import Intermediate

strict_unary_constructor :: (Strict s) => (s -> t) -> s -> t
strict_unary_constructor ctor a = let a' = strict a in seq a' (ctor a')
strict_binary_constructor :: (Strict r, Strict s) => (r -> s -> t) -> r -> s -> t
strict_binary_constructor ctor a b = let
				a' = strict a
				b' = strict b
				in seq a' (seq b' (ctor a' b'))
strict_trinary_constructor :: (Strict p, Strict r, Strict s) => (p -> r -> s -> t) -> p -> r -> s -> t
strict_trinary_constructor ctor a b c = let
				a' = strict a
				b' = strict b
				c' = strict c
				in seq a' (seq b' (seq c' (ctor a' b' c')))
strict_quaternary_constructor :: (Strict o, Strict p, Strict r, Strict s) => (o -> p -> r -> s -> t) -> o -> p -> r -> s -> t
strict_quaternary_constructor ctor a b c d = let
				a' = strict a
				b' = strict b
				c' = strict c
				d' = strict d
				in seq a' (seq b' (seq c' (seq d' (ctor a' b' c' d'))))
strict_quintary_constructor :: (Strict n, Strict o, Strict p, Strict r, Strict s) => (n -> o -> p -> r -> s -> t) -> n -> o -> p -> r -> s -> t
strict_quintary_constructor ctor a b c d e = let
				a' = strict a
				b' = strict b
				c' = strict c
				d' = strict d
				e' = strict e
				in seq a' (seq b' (seq c' (seq d' (seq e' (ctor a' b' c' d' e')))))

-- Continue if no part of the data structure is bottom.
class Strict t where
    strict :: t -> t

instance Strict Integer where
    strict n = n
instance Strict Int where
    strict n = n
instance Strict Char where
    strict ch = ch
instance Strict Bool where
    strict tf = tf
instance Strict a => Strict [a] where
    strict (a : more) = strict_binary_constructor (:) a more
    strict [] = []
instance Strict a => Strict (Maybe a) where
    strict m = case m of
	Just x -> strict_unary_constructor Just x
	Nothing -> Nothing
instance (Strict a, Strict b) => Strict (Either a b) where
    strict e = case e of
	Left x -> strict_unary_constructor Left x
	Right y -> strict_unary_constructor Right y
instance (Strict a, Strict b) => Strict (a, b) where
    strict (x,y) = strict_binary_constructor (,) x y
instance (Strict a, Strict b, Strict c) => Strict (a, b, c) where
    strict (x,y,z) = strict_trinary_constructor (,,) x y z

instance Strict Term_ where
    strict x = case x of
	Coabstract_ str stmt stack_size -> strict_trinary_constructor Coabstract_ str stmt stack_size
	Error_ msg -> strict_unary_constructor Error_ msg
	Builtin_ v -> strict_unary_constructor Builtin_ v
	Record_ ts -> strict_unary_constructor Record_ ts
	Enumerator_ m n t -> strict_trinary_constructor Enumerator_ m n t
	Plan_ c -> strict_unary_constructor Plan_ c
	Exit_ orig -> strict_unary_constructor Exit_ orig
	Do_ orig name t -> let
			orig' = strict orig
			name' = strict name
			in seq name' $ Do_ orig' name' t
	Be_ orig c -> strict_binary_constructor Be_ orig c
	Bind_ orig o1 o2 d r -> strict_quintary_constructor Bind_ orig o1 o2 d r
	Variable_ num -> strict_unary_constructor Variable_ num
	Let_ e t -> strict_binary_constructor Let_ e t
instance Strict Statement_ where
    strict (Cut_ origin w c) = strict_trinary_constructor Cut_ origin w c
instance Strict Coterm_ where
    strict x = case x of
	Case_ cs -> strict_unary_constructor Case_ cs
	Selector_ m n c -> strict_trinary_constructor Selector_ m n c
	Use_ t -> strict_unary_constructor Use_ t
	Expose_ c id next -> strict_unary_constructor Expose_ c id next
	Covariable_ num -> strict_unary_constructor Covariable_ num
	Abstract_ var stmt stack_size -> strict_trinary_constructor Abstract_ var stmt stack_size
	Function_ orig name f -> let
			orig' = strict orig
			name' = strict name
			in seq name' $ Function_ orig' name' f
	Colet_ e c -> strict_binary_constructor Colet_ e c

instance Strict BuiltinValue where
    strict v = case v of
	BuiltinInteger n -> strict_unary_constructor BuiltinInteger n
	BuiltinString str -> strict_unary_constructor BuiltinString str
	BuiltinBoolean b -> strict_unary_constructor BuiltinBoolean b
	BuiltinList ts -> strict_unary_constructor BuiltinList ts
	BuiltinColist cs -> strict_unary_constructor BuiltinColist cs
