{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Unify (
	TypeInContext(TypeInContext), addToTypeContext, listToTypeContext, TypeContext, drop_type_context,
	unify, prep,
	threadUTRefTransform, substituteTypeName,
	typeMatchesOf, Typed,
	run_with_typevars
	) where
import Monad
import List
import Maybe
import Data.Map hiding(map, lookup, union, delete, null, (\\))
import qualified Data.Map
import Data.IORef
import Control.Monad.State
import IO
import Char

import Type
import Kind hiding (Unifier, relink)
import qualified Kind
import Basic
import Position

-- The first member ("kind") of LambdaT, ExistsT, and ForallT are typ rather than k so that
-- they can hold a VarT which is a proxy for the named type and which can be unified with
-- other named types.

-- YVT Type is the type of a function which takes itself as its first argument.
-- Note that it is not the type of Y (or fix), and is not for source-level use.
-- That would be an infinite type if we didn't use YVT.  
--    YVT t == (YVT t) -> t         YNT t == t <- (YNT t)
-- See example in Pierce, "Recursive Values from Recursive Types".
-- The YXT_ types are similar, but work with lowered types.

original_type :: Tp Type -> Tp Type
original_type tp = case tp of
	AndT CofunctionOrigin [Type _ _ (ApplicationT (Type _ _ NotT) argt), resultt] -> CofunctionT argt resultt
	ApplicationT (Type _ _ NotT)
	    (Type _ _ (AndT FunctionOrigin
			    [argt,
			    (Type _ _ (ApplicationT (Type _ _ NotT) resultt))]))
		    -> FunctionT argt resultt
	ApplicationT (Type _ sp (NameT "IO" "V")) resultt
		    -> ApplicationT (ctorT sp "IOV") resultt
	ApplicationT (Type _ _ NotT)
	    (Type _ _ (ApplicationT (Type _ sp (NameT "IO" "S")) resultt))
		    -> ApplicationT (ctorT sp "ION") (notT sp resultt)
	ApplicationT (Type _ _ NotT)
	    (Type _ _ (ApplicationT (Type _ _ (NameT "IO" "N"))
		(Type _ sp (ApplicationT (Type _ _ NotT) resultt))))
		    -> ApplicationT (ctorT sp "ION") resultt
	otherwise -> tp

original_utref :: (Functor m, MonadIO m) => Tp UTRef -> m (Tp UTRef)
original_utref tp = case tp of
	AndT CofunctionOrigin [argref, resref] -> do
		UType _ _ argtp <- liftIO $ readIORef argref
		case argtp of
			ApplicationT ctorref aref -> do
				UType _ _ nottp <- liftIO $ readIORef ctorref
				case nottp of
					NotT -> return $ CofunctionT aref resref
					otherwise -> return tp
			otherwise -> return tp
	ApplicationT ctorref bref -> do
		UType _ _ ctortp <- liftIO $ readIORef ctorref
		UType _ sp btp <- liftIO $ readIORef bref
		case (ctortp, btp) of
		    (NotT, AndT FunctionOrigin [argref, rref]) -> do
			UType _ _ rtp <- liftIO $ readIORef rref
			case rtp of
			    ApplicationT ctorref resref -> do
				UType _ _ nottp <- liftIO $ readIORef ctorref
				case nottp of
					NotT -> return $ FunctionT argref resref
					otherwise -> return tp
			    otherwise -> return tp
		    (NameT "IO" "V", _) -> do
			tref <- uIOVT (UseLowerIOType False) sp bref
			UType _ _ tp' <- liftIO $ readIORef tref
			return tp'
		    (NotT, ApplicationT ioctorref etref) -> do
			UType _ _ ioctortp <- liftIO $ readIORef ioctorref
			UType _ _ etp <- liftIO $ readIORef etref
			case (ioctortp, etp) of
				(NameT "IO" "S", _) -> do
				    notc_tref <- uctorT_ sp NotT
				    not_tref <- liftIO $ newIORef (uprop sp (ApplicationT notc_tref etref))
				    tref <- uIONT (UseLowerIOType False) sp not_tref
				    UType _ _ tp' <- liftIO $ readIORef tref
				    return tp'
				(NameT "IO" "N", ApplicationT notctorref resultref) -> do
				    UType _ _ notctortp <- liftIO $ readIORef notctorref
				    case notctortp of
					NotT -> do
					    tref <- uIONT (UseLowerIOType False) sp resultref
					    UType _ _ tp' <- liftIO $ readIORef tref
					    return tp'
					otherwise -> return tp
				otherwise -> return tp
		    otherwise -> return tp
	otherwise -> return tp

instance Show (Tp Type) where
    show tp = show_tp $ fmap (pshow_type ShowSpan True) (original_type tp)
instance KShow Type where
    kshow limit ds ks (Type k sp tp) = show_tp (fmap (pshow_type (nest ds) ks) (original_type tp)) ++ (if ks then ":" ++ show k else "")
show_full_type :: PString -> String
show_full_type (PString prec str) = str
show_quaternary_type (PString prec str) = if prec < QuaternaryType then "(" ++ str ++ ")" else str
show_tertiary_type (PString prec str) = if prec < TertiaryType then "(" ++ str ++ ")" else str
show_secondary_type (PString prec str) = if prec < SecondaryType then "(" ++ str ++ ")" else str
show_primary_type (PString prec str) = if prec < PrimaryType then "(" ++ str ++ ")" else str
data TypePrecedence = FullType | QuaternaryType | TertiaryType | SecondaryType | PrimaryType
	deriving (Eq,Ord)
data PString = PString TypePrecedence String

pshow_type :: SpanDisplay -> Bool -> Type -> PString
pshow_type ds dk (Type k sp tp) = PString prec $ s ++ (if dk then ":" ++ show k' else "")
    where
	tp' = original_type tp
	tp'' = fmap (pshow_type ds dk) tp'
	k' = fmap show k
	PString prec s = pshow ds dk tp''
show_tp tp = let PString prec s = pshow ShowSpan True tp in s
pshow :: SpanDisplay -> Bool -> Tp PString -> PString
pshow ds ks t = case t of
	VarT id ref -> PString PrimaryType $ show id
	NameT nm _ -> PString PrimaryType nm
	AndT origin ts -> PString PrimaryType $ "(" ++ "&" ++ concatMap (\t -> show_full_type t ++ "&") ts ++ ")"
	OrT ts -> PString PrimaryType $ "(|" ++ concatMap (\t -> show_full_type t ++ "|") ts ++ ")"
	NotT -> PString PrimaryType $ "~"
	NoT -> PString PrimaryType $ "_"
	FunctionT arg result -> PString QuaternaryType $ show_tertiary_type arg ++ "->" ++ show_quaternary_type result
	CofunctionT arg result -> PString QuaternaryType $ show_tertiary_type arg ++ "<+" ++ show_quaternary_type result
	ApplicationT constructor arg -> let
		c = show_secondary_type constructor
		a = show_primary_type arg
		starts_with_alphanum "" = False
		starts_with_alphanum (c:_) = isAlphaNum c
		run_on = starts_with_alphanum (reverse c) && starts_with_alphanum a
	    in
		PString SecondaryType $ c ++ (if run_on then " " else "") ++ a
	LambdaT kind nm typ -> PString FullType $ "\\" ++ nm ++ (if ks then ":" ++ show_full_type kind else "") ++ ":>" ++ show_full_type typ
	YNT tp -> PString SecondaryType $ "YN " ++ show_primary_type tp
	YVT tp -> PString SecondaryType $ "YV " ++ show_primary_type tp
	ExistsT kind nm tp -> PString FullType $ "some " ++ nm ++ (if ks then ":" ++ show_full_type kind else "") ++ "." ++ show_full_type tp
	ForallT kind nm tp -> PString FullType $ "all " ++ nm ++ (if ks then ":" ++ show_full_type kind else "") ++ "." ++ show_full_type tp
	UnsatisfiedT matchmaker string -> PString FullType "unsatisfied constraint in type matching"

instance KindParameterized UTRef where
    kindMatchesOf kmap tref = do
	typ@(UType kind sp tp) <- liftIO $ readIORef tref
	local_matches <- case tp of
	    VarT ids ref -> return []
	    NameT nm _ -> return $ maybe ( {- trace ("Type name " ++ nm ++ " is not bound.") -} [])
			    ( {- trace ("Type name " ++ nm ++ " _is_ bound.") -} (\k -> [(k, tref)]))
			    (Data.Map.lookup nm kmap)
	    NotT -> do
			    predicate_kind <- prep_kind (Kind (Predicate (Kind Proposition) (Kind Proposition)))
			    return [(tref, predicate_kind)]
	    NoT -> return []
	    ApplicationT ctref atref -> do
			    UType constr csp _ <- liftIO $ readIORef ctref
			    at_kind <- extract_kind atref
			    t_kind <- extract_kind tref
			    predicate_kind <- prep_kind (Kind (Predicate at_kind t_kind))
			    return [(ctref, predicate_kind)]
	    LambdaT param_kind nm body_tref -> do
			    body_tref_kind <- extract_kind body_tref
			    param_kind' <- extract_kind param_kind
			    predicate_kind <- prep_kind (Kind (Predicate param_kind' body_tref_kind))
			    return [(tref, predicate_kind)]
	    otherwise -> do
			    proposition_kind <- uprop_kind
			    return [(tref, proposition_kind)]
	other_matches <- kindMatchesOf kmap tp
	finished_local_matches <- mapM (finish typ) local_matches
	return $ finished_local_matches ++ other_matches
      where
	finish typ (tp1,tp2) = do
		typ_str <- do_kshow default_type_limit NoShowSpan True tref
		return (tp1,tp2,typ_str)

threadUTRefTransform :: MonadIO m => (UTRef -> m UTRef) -> UTRef -> m UTRef
threadUTRefTransform f utref = do
	    UType k sp tp <- liftIO $ readIORef utref
	    tp' <- fmapM (threadUTRefTransform f) tp
	    k' <- fmapM (threadUTRefTransform f) k
	    utref' <- liftIO $ newIORef (UType k' sp tp')
	    result <- f utref'
	    return result

-- !! Note this is not a correct implementation because it may go too deeply
-- when the name is hidden by another declaration of the same name.
-- That won't be very difficult to fix.
substituteTypeName :: (Functor m, MonadIO m) => String -> UTRef -> UTRef -> m UTRef
substituteTypeName name subst_tp = substitute_in_ref
    where
	substitute_in_ref :: (Functor m, MonadIO m) => UTRef -> m UTRef
	substitute_in_ref utref = do
	    UType k sp tp <- liftIO $ readIORef utref
	    case tp of
		NameT name' _ | name' == name -> return subst_tp
		otherwise -> do
		    tp' <- fmapM substitute_in_ref tp
		    utref' <- liftIO $ newIORef (UType k sp tp')
		    case tp of
			VarT ids tref -> do -- tref points to a VarT from the original clique.
			    cc "0012" tref
			    tvars <- fmap (tref :) (more_tvars "2" tref tref)
			    if not (utref' `elem` tvars) then
				    relink (last tvars) utref'
				else return ()
			    cc "12" utref'
			    cc "012" tref
			otherwise -> return ()
		    return utref'

instance KindParameterized (Tp UTRef) where
    kindMatchesOf kmap tp = finish $ case tp of
	    VarT ids ref -> return []
	    NameT nm _ -> return []
	    NoT -> return []
	    AndT _ ts -> mapM is_prop ts
	    OrT ts -> mapM is_prop ts
	    NotT -> return []
	    FunctionT from to -> do
		    from_is <- is_prop from
		    to_is <- is_prop to
		    return [from_is, to_is]
	    ApplicationT constr arg ->  return []
	    LambdaT kind nm body -> return []
	    YVT tp -> is_one_prop tp
	    YNT tp -> is_one_prop tp
	    CofunctionT from to -> do
		    from_is <- is_prop from
		    to_is <- is_prop to
		    return [from_is, to_is]
	    ExistsT kind nm tp -> return []
	    ForallT kind nm tp -> return []
	    UnsatisfiedT matchmaker msg -> return []
	    -- (UnexistsT stp t1, ExistsT nm t2) ->
	    --	GenericTypeTerm nm (TM ttp t) -> ForallT nm ttp
	    --	SpecificTerm stp (TO ttp t) -> UnsatisfiedT needs ForallT
	    --	PackageTerm stp (TM ttp t) -> UnsatisfiedT needs ExistsT
	    --	AbstractTypeTerm nm (TO ttp t) -> ExistsT nm ttp
    {-
       A Generic/AbstractTypeTerm introduces a name with kind matching 
       what it has in the type.

       A SpecificTerm/PackageTerm must occur with a quantified type,
       and the substituted type's kind must match the quantifier kind.

       Maybe ForallT should be ForallT Kind Type and the substitution
       handled somewhere else.
    -}

	where
	    finish get_local_matches = do
		local_matches <- get_local_matches
		other_matcheses <- mapM (kindMatchesOf kmap') (typeChildrenOf tp)
		finished_local_matches <- mapM finish' local_matches
		return $ finished_local_matches ++ concat other_matcheses
	    finish' (tp1,tp2) = do
		tp_str <- do_kshow default_type_limit NoShowSpan True tp
		return (tp1,tp2,tp_str)
	    kmap' = case tp of
		    ExistsT kind nm tp -> Data.Map.insert nm kind kmap
		    ForallT kind nm tp -> Data.Map.insert nm kind kmap
		    LambdaT kind nm typ -> Data.Map.insert nm kind kmap
		    otherwise -> kmap
	    is_prop tref = do
		    proposition_kind <- uprop_kind
		    return (tref, proposition_kind)
	    is_one_prop tref = do
		    match <- is_prop tref
		    return [match]
instance FunctorM Tp where
    fmapM f tp = case tp of
	VarT ids ref -> return $ VarT ids ref
	NameT nm x -> return $ NameT nm x
	NoT -> return $ NoT
	AndT origin ts -> do
			ts' <- mapM f ts
			return $ AndT origin ts'
	OrT ts -> do	ts' <- mapM f ts
			return $ OrT ts'
	NotT -> return NotT
	FunctionT a r -> do
			a' <- f a
			r' <- f r
			return $ FunctionT a' r'
	CofunctionT a r -> do
			a' <- f a
			r' <- f r
			return $ CofunctionT a' r'
	ApplicationT c a -> do
			c' <- f c
			a' <- f a
			return $ ApplicationT c' a'
	LambdaT k nm t -> do
			k' <- f k
			t' <- f t
			return $ LambdaT k' nm t'
	YNT t -> do	t' <- f t
			return $ YNT t'
	YVT t -> do	t' <- f t
			return $ YVT t'
	ExistsT k nm t -> do
			k' <- f k
			t' <- f t
			return $ ExistsT k' nm t'
	ForallT k nm t -> do
			k' <- f k
			t' <- f t
			return $ ForallT k' nm t'
	UnsatisfiedT matchmaker msg -> return $ UnsatisfiedT matchmaker msg

-- head of list becomes root (not top) of stack
listToTypeContext :: [(VarName,(Way,UTRef))] -> TypeContext
listToTypeContext associations = foldr combine empty (reverse associations)
    where
	combine (name, way_type) tc = addToTypeContext tc name way_type

addToTypeContext :: TypeContext -> VarName -> (Way, UTRef) -> TypeContext
addToTypeContext tc name (way, tp) = Data.Map.insert name (way, (TypeInContext tp tc)) tc

-- Ugly code to look for type variable in type context.
findTypeInContext :: TypeContext -> UTRef -> StateT [TypeId] IO (Maybe VarName)
findTypeInContext tc tref = do
	maybe_found <- foldM find Nothing (toList tc)
	case maybe_found of
	    Just (name, tref') -> case Data.Map.lookup name tc of
		Just (way', TypeInContext tref'' _) ->
					  if tref'' == tref' then return (Just name)
						else return Nothing
		otherwise -> return Nothing
	    otherwise -> return Nothing
    where
	find found (name, (way, TypeInContext tref' _)) = case (found, way) of
	    (Nothing, ByType) -> do
			    t' <- liftIO $ readIORef tref'
			    case t' of
				UType _ _ (VarT _ _) -> do
				    tvars <- fmap (tref' :) (more_tvars "1" tref' tref')
				    if tref `elem` tvars then return (Just (name, tref'))
					else return Nothing
				otherwise -> if tref == tref' then return (Just (name, tref'))
					else return Nothing
	    otherwise -> return found

instance (Functor m, MonadIO m) => MonadicKShow m TypeInContext
    where
	do_kshow limit ds dk (TypeInContext v _) = do_kshow limit (nest ds) dk v

class Typed t where
	typeMatchesOf :: MonadModel -> UseIOType -> UseFunctionTypes -> TypeContext -> t
		-> StateT [TypeId] IO [(TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)]
		-- The String is to identify where the match arose.
		-- Each Maybe Way indicates whether the type is on an expression
		-- which admits removal of quantifiers.  If so, 
		-- ByValue indicates that a ForallT quantifier may be removed.
		-- ByName indicates that an ExistsT quantifier may be removed.

instance Parameterized Type where
    -- namesOf :: p -> [VarName]  -- exposed names
    namesOf (Type k sp tp) = foldTree names tp
	where
	    names tp sublist = case tp of 
		    NameT nm _ -> [nm]
		    ExistsT k nm tp -> nm `delete` sublist'
		    ForallT k nm tp -> nm `delete` sublist'
		    otherwise -> sublist'
		where
		    sublist' = foldr union [] sublist
    -- usedOf :: p -> [VarName]   -- all names
    usedOf (Type k sp tp) = foldTree used tp
	where
	    used tp sublist = case tp of
		    NameT nm _ -> [nm]
		    NoT -> ["_"]
		    ExistsT k nm tp -> [nm] `union` sublist'
		    ForallT k nm tp -> [nm] `union` sublist'
		    otherwise -> sublist'
		where
		    sublist' = foldr union [] sublist

instance Substitutable Type Type where
    substitute nm sub subee@(Type k sp subee_tp) = case subee_tp of
	NameT nm' _
		| nm == nm' -> sub
		| otherwise -> subee
	ExistsT k' nm' tp
		| nm == nm' -> subee
		| otherwise -> Type k sp $ ExistsT k' nm' $ more tp
	ForallT k' nm' tp 
		| nm == nm' -> subee
		| otherwise -> Type k sp $ ForallT k' nm' $ more tp
	otherwise -> Type k sp $ fmap more subee_tp
      where
	more = substitute nm sub

instance Functor Tp where
    fmap xfrm tp = case tp of
	AndT origin ts -> AndT origin $ map xfrm ts
	OrT ts -> OrT $ map xfrm ts
	NotT -> NotT
	FunctionT arg result -> FunctionT (xfrm arg) (xfrm result)
	CofunctionT arg result -> CofunctionT (xfrm arg) (xfrm result)
	ApplicationT constr arg -> ApplicationT (xfrm constr) (xfrm arg)
	LambdaT k nm t -> LambdaT (xfrm k) nm (xfrm t)
	YNT t -> YNT (xfrm t)
	YVT t -> YVT (xfrm t)
	ExistsT k nm' t -> ExistsT (xfrm k) nm' $ xfrm t
	ForallT k nm' t -> ForallT (xfrm k) nm' $ xfrm t
	UnsatisfiedT matchmaker str -> UnsatisfiedT matchmaker str
	NoT -> NoT
	NameT nm x -> NameT nm x
	VarT ids ref -> VarT ids ref

run_with_typevars :: StateT [TypeId] IO t -> IO t
run_with_typevars action = do
	(val, typevars') <- runStateT action [1..]
	return val
	
type_vars_of :: Type -> [UTRef]
type_vars_of (Type k sp tp) = foldTree tvo tp
    where
	tvo t sublist = case t of
		VarT ids ref -> [ref]
		otherwise -> foldr union [] sublist

-- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == --
{-
Unification is driven by the source code.  Each source code construct implies
constraints between the type of the expression, and the types of its
subexpressions.  Where there's a name reference, it also brings in
the type of that name.  In the following, a constraint is called a
"type match", because it relates one type to another. 

The types in a type match may be omitted in the source code or may
be otherwise underspecified, in which case a VarT acts as a placeholder
that needs to be filled in based on matches.

Each type match is processed with the goal of checking that the two types
found in the source correspond correctly to one another.  When the
processing runs into a VarT, it generates a "unifier" indicating that
that type variable is to be replaced with the corresponding type
expression from the other side of the type match.  Each generated
unifier is applied to the source code so as to elaborate the type of each
subexpression.  Only when all unifiers have been applied and the type
matches have been re-processed without any mismatches is the source code
validated.

The original implementation would process one unifier per pass through
the source code.

In the faster implementation, each time a unifier is found, it is applied
immediately to the types throughout the source code as well as to the map
of names that are in scope.

Applying the unifier consists of replacing each corresponding occurrence of
VarT with the new type. Hence each subtype is represented using an IORef cell.
When several cells indicate the same variable, the cells are linked in
a circular list.
Moreover, the occurrences must be chained together for modification, with
the chain available via the VarT.  The chain is equivalent to a circular,
internally-linked list of IORefs.  Unification will need to be careful
about circular references.

-}

-- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == --

prep :: Type -> StateT [TypeId] IO UTRef
prep (Type k sp tp) = do
	tp' <- fmapM prep tp
	k' <- fmapM prep_kind k
	ref <- liftIO $ newIORef (UType k' sp tp')
	case tp' of
		NoT -> do
			mv <- liftState get
			case mv of
			    (v:_) -> do
				liftState (modify tail)
				liftIO $ writeIORef ref (UType k' sp (VarT v ref))
				cc "1" ref
			    otherwise -> return ()
		VarT ids tref -> do -- tref points to a VarT from the original clique.
		    cc "0013" tref
		    tvars <- fmap (tref :) (more_tvars "21" tref tref)
		    if not (ref `elem` tvars) then
			    relink (last tvars) ref
			else return ()
		    cc "13" ref
		    cc "013" tref
		_ -> return ()
	post_prep_kind ref k'
	return ref

extract :: UTRef -> IO Type
extract ref = do
	(UType k sp tp) <- readIORef ref
	tp' <- fmapM extract tp
	k' <- fmapM extract_kind k
	return (Type k' sp tp')

do_pshow :: (Functor io, MonadIO io) => Integer -> SpanDisplay -> Bool -> UTRef -> io PString
do_pshow limit ds dk tp_ref = if limit > 0 then do
			(UType k sp tp) <- liftIO $ readIORef tp_ref
			tp' <- original_utref tp
			tp'' <- fmapM (do_pshow (limit-1) ds dk) tp'
			k' <- fmapM (do_kshow (limit-1) ds dk) k
			let PString prec s = pshow ds dk tp''
			return $ PString prec $ s ++ (if dk then ":" ++ show k' else "")
    else
	return $ PString PrimaryType "..."

instance (Functor io, MonadIO io) => MonadicKShow io UTRef
    where
	-- do_kshow :: SpanDisplay -> Bool -> UTRef -> io String
	do_kshow limit ds dk tp_ref = do
		PString prec s <- do_pshow limit ds dk tp_ref
		return s

instance (Functor m, Monad m, MonadIO m) => MonadicKShow m (Tp UTRef)
    where
	do_kshow limit ds dk reffer = do
				val <- fmapM (do_pshow limit (nest ds) dk) reffer
				return (show_tp val)

data UnifierResult = GoodUnifier Unifier | BadUnifier Type Type
data Unifier
	= Unifier TypeInContext (StateT [TypeId] IO TypeInContext) String String
	| KindUnifier UTRef UTRef String
	| Unsatisfied (StateT [TypeId] IO (Maybe (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)) ) String
instance (MonadIO m) => MonadicShow m Unifier where
    do_show uni = case uni of
	Unifier id tpf display wher -> return (display ++ ", at " ++ wher)
	KindUnifier k1 k2 str -> do
		k1_str <- do_show k1
		k2_str <- do_show k2
		return $ "(kind mismatch for " ++ k1_str ++ ", " ++ k2_str ++ ", "  ++ str ++ ")"
	Unsatisfied matchmaker str -> return $ "(unsatisfied: " ++ str ++ ")"

-- The limit affects the maximum size of *types* that are handled.
unify :: (Typed (e UTRef), FunctorM e, MonadicTShow (StateT [TypeId] IO) (e UTRef),
         KindParameterized (e UTRef)
         )
	=> Maybe Handle -> Bool -> Bool -> Integer
		-> MonadModel -> UseIOType -> UseFunctionTypes
		-> [(VarName, (Way, Type))] -> (e Type) -> StateT [TypeId] IO (e Type)
unify verbose verbose_kinds continue limit mm uio ufn builtins stmt = do
	stmt' <- fmapM prep stmt
	case verbose of
	    Just h -> do
		stmt'str <- do_tshow NoShowSpan True verbose_kinds stmt'
		liftIO $ hPutStrLn h $ "Unifying types, the prepped expression is: " ++ stmt'str
	    Nothing -> return ()
	builtins_list' <- mapM prep_builtin builtins
	let builtins' = listToTypeContext builtins_list'
	unify_kinds verbose (if verbose_kinds then verbose else Nothing)
			limit mm uio ufn Nothing builtins' stmt'
	case verbose of
	    Just h -> do
		stmt'str <- do_tshow NoShowSpan True verbose_kinds stmt'
		liftIO $ hPutStrLn h $ "Unifying types, the expression with unified kinds is: "
						++ stmt'str
	    Nothing -> return ()
	type_matches <- typeMatchesOf mm uio ufn builtins' stmt'
	deferrals <- unify_matches ufn verbose verbose_kinds
				type_matches [] [] continue limit builtins'
	case deferrals of
	    [] -> return ()
	    ((Unsatisfied matchmaker message):_) -> do
		case verbose of
		    Just h -> do
			liftIO $ hPutStrLn h $ "Unsatisfied unifier."
			stmt'str <- do_tshow NoShowSpan True verbose_kinds stmt'
			liftIO $ hPutStrLn h $ "Final inferenced expression is: " ++ stmt'str
		    Nothing -> return ()
		die message
	    _ -> die "Unexpected deferred unifier."

	-- Double check.
	modify (const [])
	type_matches' <- typeMatchesOf mm uio ufn builtins' stmt'
	m_error <- double_check_unified ufn verbose verbose_kinds
				type_matches' limit builtins'
	case m_error of
	    Just message -> do
		case verbose of
		    Just h -> do
			liftIO $ hPutStrLn h "Incomplete unification."
			stmt'str <- do_tshow NoShowSpan True verbose_kinds stmt'
			liftIO $ hPutStrLn h $ "Final inferenced expression is: " ++ stmt'str
		    Nothing -> return ()
		die message
	    Nothing -> return ()

	liftIO (fmapM extract stmt')
    where
	prep_builtin :: (VarName, (Way, Type)) -> StateT [TypeId] IO (VarName, (Way, UTRef))
	prep_builtin (nm, (way, tp)) = do
		tpref <- prep tp
		return (nm, (way, tpref))

-- A deferral is unlike a match, in that it will continue to be attempted until
-- it can analyze its operator for type matches.
-- Progress is noted whenever there's a change to the type of an operator,
-- so that deferrals may be re-attempted.  Deferrals are classified as
-- "hopeless" if there has been no progress since the previous time they were attempted.

-- If a conflict is found, unify_matches exits rather than returns.
-- Returns a list of unsatisfied unifiers.
unify_matches :: UseFunctionTypes -> Maybe Handle -> Bool -> [(TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)]
		      -> [Unifier] -> [Unifier] -> Bool -> Integer
		      -> Map VarName (Way, TypeInContext)
                      -> StateT [TypeId] IO [Unifier]
unify_matches ufn verbose verbose_kinds matches hopeful_deferrals hopeless_deferrals continue limit builtins = do
    (v:_) <- get
    -- aside verbose ("unify_matches with builtins of " ++ show (keys builtins))
    maybe_more <- case (matches, hopeful_deferrals) of
	(match:matches',_) -> do
	    aside verbose $ "unification processing a match"
	    return $ Just (Just match, matches', hopeful_deferrals, hopeless_deferrals)
	([], (deferral@(Unsatisfied matchmaker msg): deferrals')) -> do
		aside verbose $ "unification processing a deferral: \"" ++ msg ++ "\""
		maybe_match <- matchmaker
		case maybe_match of
		    Just match@(TypeInContext t1 _,TypeInContext t2 _,msgf) -> do
			match_str1 <- do_kshow limit TopShowSpan verbose_kinds t1
			match_str2 <- do_kshow limit TopShowSpan verbose_kinds t2
			msg <- msgf verbose_kinds match_str1 match_str2
			aside verbose $ "unification processing a deferred match:\n"
						  ++ "(" ++ match_str1 ++ ", " ++ match_str2 ++ "," ++ msg ++ ")"
			return $ Just (Just match, [], deferrals', hopeless_deferrals)
		    Nothing -> do
				aside verbose $ "not ready deferral: \"" ++ msg ++ "\""
				return $ Just (Nothing, [], deferrals', deferral:hopeless_deferrals)
	([], (deferral: deferrals')) -> die "Unexpected deferred unifier."
	([],[]) -> return Nothing
    hd <- case maybe_more of
	Just (maybe_match, further_matches, hopeful_deferrals, hopeless_deferrals) -> do
	    (progress, more_deferrals) <- case maybe_match of
		Just match -> unify_match ufn verbose verbose_kinds match continue limit builtins
		Nothing -> return (False, [])
	    case verbose of
		Just h -> mapM_ (\deferral@(Unsatisfied _ msg) -> do
			    liftIO $ hPutStrLn h $ "queueing a new deferral: \"" ++ msg ++ "\""
			) more_deferrals
		Nothing -> return ()
	    (hopeful_deferrals', hopeless_deferrals') <-
		if progress then do
		    case verbose of
			Just h -> do
			    liftIO $ hPutStrLn h $ "unification progress"
			    let deferrals = map (\(Unsatisfied _ msg) -> msg) (hopeful_deferrals ++ hopeless_deferrals ++ more_deferrals)
			    liftIO $ hPutStrLn h $ "deferrals are: " ++ show deferrals
			Nothing -> return ()
		    return (hopeful_deferrals ++ hopeless_deferrals ++ more_deferrals, [])
		 else do
		    unless (null more_deferrals) $ aside verbose $ "unification deferred " ++ show (length more_deferrals)
									 ++ " matches"
		    return (hopeful_deferrals, hopeless_deferrals ++ more_deferrals)
	    unify_matches ufn verbose verbose_kinds further_matches
					hopeful_deferrals' hopeless_deferrals'
					continue limit builtins
	Nothing -> return hopeless_deferrals
    return hd

mkmsg :: Integer -> Bool -> (Bool -> String -> String -> StateT [TypeId] IO String) -> UTRef -> UTRef -> StateT [TypeId] IO String
mkmsg limit show_kind msgf tref1 tref2 = do
	ts1 <- liftIO $ do_kshow limit TopShowSpan show_kind tref1
	ts2 <- liftIO $ do_kshow limit TopShowSpan show_kind tref2
	msgf show_kind ts1 ts2

unify_match :: UseFunctionTypes -> Maybe Handle -> Bool -> (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String) -> Bool -> Integer -> Map VarName (Way, TypeInContext) -> StateT [TypeId] IO (Bool, [Unifier])
unify_match ufn verbose verbose_kinds (tic1@(TypeInContext tref1 ctx1), tic2@(TypeInContext tref2 ctx2), wher) continue limit builtins = do
	msg <- mkmsg limit verbose_kinds wher tref1 tref2
	aside verbose ("unify_match\nat " ++ msg {- ++ "\nwith ctx1 of " ++ show (keys ctx1) ++ "\nwith ctx2 of " ++ show (keys ctx2) -})
	(prgs, mus) <- unify2 ufn verbose verbose_kinds False False continue limit msg tic1 tic2
	case mus of
	    Nothing -> do
		t1@(UType k1 _ _) <- liftIO (readIORef tref1)
		t2@(UType k2 _ _) <- liftIO (readIORef tref2)
		msg <- mkmsg limit (k1 /= k2) wher tref1 tref2
		die ("The types cannot be unified: " ++ msg)
	    Just us -> do
		results <- mapM (apply_unifier ufn verbose verbose_kinds continue limit builtins) us
		let (progresses, deferralses) = unzip results
		let deferrals = concat deferralses
		let progress = or progresses || prgs
		return (progress, deferrals)

-- Returns a boolean indicating progress, and a list of Unsatisfied unifiers to try again later.
apply_unifier :: UseFunctionTypes -> Maybe Handle -> Bool -> Bool -> Integer -> Map VarName (Way, TypeInContext) -> Unifier -> StateT [TypeId] IO (Bool, [Unifier])
apply_unifier ufn verbose verbose_kinds continue limit builtins u = do
  aside verbose "applying a unifier"
  case u of
    KindUnifier k1 k2 str -> do
	    aside verbose "applying a kind unifier"
	    u_str <- do_show u
	    progress <- apply_kind_unifier (if verbose_kinds then verbose else Nothing) (Kind.Unifier k1 k2 str)
	    return (progress, [])
    Unsatisfied matchmaker str -> do
	    aside verbose $ "applying an unsatisfied unifier for " ++ str
	    maybe_new_match <- matchmaker
	    aside verbose "applying an unsatisfied unifier with match"
	    ret <- case maybe_new_match of
		Just match -> unify_match ufn verbose verbose_kinds match continue limit builtins
		Nothing -> return (False, [u])
	    aside verbose "applied an unsatisfied unifier"
	    return ret	    
    Unifier tic@(TypeInContext tvar_ref tvar_ctx) trefgen str wher -> do
	    aside verbose "applying a type unifier"
	    tvar <- liftIO $ readIORef tvar_ref
	    aside verbose "applying a type unifier, about to generate tref"
	    tic_with_assigned_typevars@(TypeInContext tref_with_assigned_typevars _) <- trefgen
	    case verbose of
		Just h -> liftIO $ do
			tvar_str <- do_kshow limit NoShowSpan False tvar_ref
			str_for_with_assigned <- do_kshow limit NoShowSpan False tref_with_assigned_typevars
			hPutStrLn h ("Applying unifier to variable " ++ tvar_str ++ ": " ++ str_for_with_assigned)
		Nothing -> return ()
	    tval_with_assigned_typevars <- liftIO $ readIORef tref_with_assigned_typevars
	    case tvar of
		UType k' sp' (VarT ids' tvar') -> do
		    tvars <- fmap (tvar_ref :) (more_tvars "3" tvar_ref tvar_ref)
		    case tval_with_assigned_typevars of
			UType k' sp' (VarT ids' tvar') -> do
			    link_two_variables "1" verbose tvar_ref tref_with_assigned_typevars
			    return (True, [])
			otherwise -> do
			    mapM_ (\tvar -> liftIO $ writeIORef tvar tval_with_assigned_typevars) tvars

			    -- A variable has been updated; unify some more.
			    m <- unify_match ufn verbose verbose_kinds
				(tic, tic_with_assigned_typevars, \_ -> \t1-> \t2-> return $ "Types " ++ t1 ++ " and " ++ t2 ++ " must match, as a consequence of the following. " ++ wher)
				continue limit builtins
			    return m
		otherwise -> do -- A variable has been updated; unify some more.
		    m <- unify_match ufn verbose verbose_kinds
			(tic, tic_with_assigned_typevars, \_ -> \t1-> \t2-> return $ "Types " ++ t1 ++ " and " ++ t2 ++ " must match, as a consequence of the following. " ++ wher)
			continue limit builtins
		    return m

same_tref :: (Functor m, MonadIO m) => UTRef -> UTRef -> m Bool
same_tref tref1 tref2 = do
    if tref1 == tref2 then return True
	else do
	    more <- more_tvars "4" tref1 tref1
	    return $ tref2 `elem` more

-- Return a list of all refs in the list up to, but not including, the to_tvar.
more_tvars :: (Functor m, MonadIO m) => String -> UTRef -> UTRef -> m [UTRef]
more_tvars msg to_tvar from_tvar = safe_more_tvars True from_tvar to_tvar from_tvar
    where
	safe_more_tvars pause stopper to_tvar from_tvar = do
	    this_var <- liftIO $ readIORef from_tvar
	    case this_var of
		UType k sp (VarT ids tvarref) -> if tvarref == to_tvar then return []
				    else do
					stopper' <- if pause then return stopper
					    else do
						    UType k sp (VarT ids next) <- liftIO $ readIORef stopper
						    if tvarref == next then do
								die $ "internal error: Entry into cycle at more_tvars " ++ msg
							else return ()
						    return next
					more <- safe_more_tvars (not pause) stopper' to_tvar tvarref
					return (tvarref : more)
		UType (VarKind _ _) _ _ -> die "unified a non-type-variable - kind variable"
		UType _ _ something -> do
				someshow <- do_kshow default_type_limit NoShowSpan False something
				die $ "unified a non-type-variable " ++ someshow ++ " in more_tvars " ++ msg

-- Modify the link of the from_var to point to 'to'.
relink :: (MonadIO m) => UTRef -> UTRef -> m ()
relink from_var to = do
    UType k sp (VarT ids tvar) <- liftIO $ readIORef from_var
    liftIO $ writeIORef from_var (UType k sp (VarT ids to))

get_id tvar = do
    UType k sp (VarT id next_tvar) <- liftIO $ readIORef tvar
    return id
set_id id tvar = do
    UType k sp (VarT _ next_tvar) <- liftIO $ readIORef tvar
    liftIO $ writeIORef tvar (UType k sp (VarT id next_tvar))
    cc "6" tvar

instance (KindParameterized t, Typed t, MonadicTShow (StateT [TypeId] IO) t) =>  KindParameterizedAugmented t
    where
	augmentedKindMatchesOf verbose_types verbose_kinds limit mm uio ufn builtins defs e = do
		type_matches <- typeMatchesOf mm uio ufn builtins e
		more_matches' <- mapM unify_for_mismatches type_matches
		let more_matches = concat more_matches'
		base_matches <- kindMatchesOf defs e
		return $ base_matches ++ more_matches
	    where
		reduce_to_kinds ((Type k1 sp1 tp1), (Type k2 sp2 tp2), msg) = (k1, k2, msg)
		-- more_matches = map reduce_to_kinds type_matches
		unify_for_mismatches (typ1, typ2, msg) = do
		    aside verbose_types $ "Unifying for mismatches" {- ++ ", " ++ msg -}
		    shown_e <- do_tshow TopShowSpan False False e
		    (_, m_unifiers) <- unify2 ufn verbose_types verbose_kinds True False True limit shown_e typ1 typ2
		    let kind_matches = 
			    maybe
			    []
			    (\unifiers -> concatMap kind_match_from unifiers)
			    m_unifiers
		    return kind_matches
		  where
		    kind_match_from uni = case uni of
			KindUnifier k1 k2 msg -> [(k1, k2, msg)]
			otherwise -> []

show_utref_list_from :: (MonadIO m) => Integer -> UTRef -> m String
show_utref_list_from n tref = do
	trefs <- get_utref_list n tref
	let trefs' = take (length (nub trefs)) trefs
	show_utref_list trefs' trefs'

get_utref_list 0 _ = return []
get_utref_list n tref = do
	UType k sp tp <- liftIO $ readIORef tref
	case tp of
		VarT ids tvarref -> do
			more <- get_utref_list (n-1) tvarref
			return (tref : more)
		otherwise -> return []

show_utref_list :: (MonadIO m) => [UTRef] -> [UTRef] -> m String
show_utref_list key utrs = do
	nextvs <- mapM (liftIO . readIORef) utrs
	let nexts = map (\(UType k sp (VarT ids next)) -> next) nextvs
	let good = nexts == (tail utrs ++ [head utrs])
	ids <- mapM get_id utrs
	let boths = zipWith (\utr id -> utr ++ show_id id) (map index utrs) ids
	return $ if good then 
	       "list " ++ show_indices utrs ++ " " ++ "{" ++ show_list (map show_id ids) ++ "}" ++ " " ++ show_list boths
	  else "non-circular list with " ++ "utrs: " ++ show_list boths ++ " nexts: " ++ show_indices nexts
    where
	index tvar = case elemIndex tvar key of
		    Just n -> show n
		    Nothing -> "?"
	show_id id = show id
	show_indices :: [UTRef] -> String
	show_indices tvs = "<" ++ show_list (map index tvs) ++ ">"
	show_int_list :: (Integral a) => [a] -> String
	show_int_list xs = sil xs
	    where
		sil [] = ""
		sil [x] = show (fromIntegral x)
		sil (x:xs) = show (fromIntegral x) ++ " " ++ sil xs
	show_list :: [String] -> String
	show_list [] = ""
	show_list [x] = x
	show_list (x:xs) = x ++ " " ++ show_list xs

link_two_variables :: String -> Maybe Handle -> UTRef -> UTRef -> StateT [TypeId] IO ()
link_two_variables msg verbose typ1 typ2 = do
			tvars1 <- fmap (typ1 :) (more_tvars ("5" ++ " " ++ msg) typ1 typ1)
			tvars2 <- fmap (typ2 :) (more_tvars ("6" ++ " " ++ msg) typ2 typ2)
			if (typ1 `elem` tvars2 || typ2 `elem` tvars1) then return () -- error "variable unified with itself"
			    else do
				-- Join the circular lists.
				relink (last tvars1) typ2
				relink (last tvars2) typ1
				cc "11" typ1
				-- Update the list of ids.
				id1 <- get_id typ1
				id2 <- get_id typ2
				mapM_ (set_id (min id1 id2)) (tvars1 ++ tvars2)
				when (id1 /= id2) $ aside verbose $ "Subsumed type variable " ++ show (max id1 id2)
									++ " into " ++ show (min id1 id2) ++ "."

-- Nothing: inconsistent
-- []: already unified
-- [(x,t):xs]: Transformation from VarT x to t is implied.

-- The context is a list of type name pairs taken from quantifiers
-- for universal and existential types that we have delved into.
-- If the lists are different, they must be accounted for when
-- unifying with named, or potentially named types.
-- In each pair, the left member corresponds to t1 and the right to t2.

-- Note that unify2 doesn't need contexts passed to it, the way u2 does,
-- because names that are not found in a context can be unified based simply
-- on their text.  But maybe it does. See context1.ambi.

unify2 :: UseFunctionTypes
	-> Maybe Handle -> Bool -> Bool -> Bool -> Bool -> Integer
	-> String -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Bool, Maybe [Unifier])
unify2 (UseLowerFunctionTypes lfn) verbose verbose_kinds kinds_only check_only continue limit wher t1 t2
	= u2 limit t1 t2
  where
    type_mismatch = (False, if kinds_only || continue then Just [] else Nothing)
    u2 :: Integer -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Bool, Maybe [Unifier])
    u2 limit tic1 tic2 = if limit /= 0 then un2 (limit - 1) tic1 tic2
				       else return type_mismatch

    un2 :: Integer -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Bool, Maybe [Unifier])
    un2 limit tic1@(TypeInContext typ1 ctx1) tic2@(TypeInContext typ2 ctx2) = do
	    UType k1 _ t1 <- liftIO $ readIORef typ1
	    UType k2 _ t2 <- liftIO $ readIORef typ2
	    case verbose of
		Just h -> do
		    t1_str <- do_kshow limit NoShowSpan verbose_kinds t1
		    t2_str <- do_kshow limit NoShowSpan verbose_kinds t2
		    liftIO $ hPutStrLn h $ "unifying" ++ (if kinds_only then " for kinds" else "")
						      ++ ": " ++ t1_str ++ " with " ++ t2_str
		Nothing -> return ()
	    case (t1,t2) of
		(NoT, _) -> (return $ (False, Just []))
		(_, NoT) -> (return $ (False, Just []))
		otherwise -> check_kinds $ un3 limit tic1 tic2
	where
	    check_kinds :: StateT [TypeId] IO (Bool, Maybe [Unifier])
			-> StateT [TypeId] IO (Bool, Maybe [Unifier])
	    check_kinds us = do
			    s_u <- show_u
			    have_match <- kinds_match typ1 typ2
			    let f us = if have_match then us
					else KindUnifier typ1 typ2 s_u : us
			    fmap (\(progress, mus) -> (progress, (fmap f) mus)) us
	    show_u :: StateT [TypeId] IO String
	    show_u = do
		    s1 <- do_show typ1
		    s2 <- do_show typ2
		    return (s1 ++ " with " ++ s2)

    un3 :: Integer -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Bool, Maybe [Unifier])
    un3 limit (TypeInContext typ1 ctx1) (TypeInContext typ2 ctx2) = do
	    maybe_typ1' <- reduce_type_application typ1 ctx1
	    let typ1' = case maybe_typ1' of
		    Just typ' -> typ'
		    Nothing -> typ1
	    maybe_typ2' <- reduce_type_application typ2 ctx2
	    let typ2' = case maybe_typ2' of
		    Just typ' -> typ'
		    Nothing -> typ2
	    un4 limit (TypeInContext typ1' ctx1) (TypeInContext typ2' ctx2)

    un4 :: Integer -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Bool, Maybe [Unifier])
    un4 limit tic1@(TypeInContext typ1 ctx1) tic2@(TypeInContext typ2 ctx2) = do
      UType k1 sp1 t1 <- liftIO $ readIORef typ1
      UType k2 sp2 t2 <- liftIO $ readIORef typ2
      case (t1,t2) of
	(UnsatisfiedT matchmaker msg, typ) -> if kinds_only then return (False, Just []) else do
				    typs <- do_show typ2
				    return $ (False, Just [Unsatisfied matchmaker $ msg ++ " The type " ++ typs ++ " was as much as could be deduced."])
	(typ, UnsatisfiedT matchmaker msg) -> if kinds_only then return (False, Just []) else do
				    typs <- do_show typ1
				    return $ (False, Just [Unsatisfied matchmaker $ msg ++ " The type " ++ typs ++ " was as much as could be deduced."])

	-- If both are variables, unify immediately so that lookup
	-- of NameT's can report non-correspondence right away.
	(VarT ids1 n1, VarT ids2 n2) -> if kinds_only then return (False, Just [])
					    else if ids1 == ids2 then
						return (False, Just [])
					    else if check_only then
						return (False, Nothing)
					    else do
						link_two_variables "2" verbose typ1 typ2
						return (True, Just [])

	-- Names must be remapped.
	(VarT ids1 n1, _) -> if kinds_only then return (False, Just [])
				else if check_only then
				    return (False, Nothing)
				else do
				    typs2 <- liftIO (do_kshow limit TopShowSpan False typ2)
				    tic2' <- remap (ctx1, sp1) tic2
				    case verbose of
					Just h -> do
					    let TypeInContext t2' _ = tic2'
					    t2str' <- do_kshow limit TopShowSpan False t2'
					    liftIO $ hPutStrLn h $ "remapped to " ++ t2str'
					Nothing -> return ()
				    return $ (True, Just [Unifier tic1 (return tic2') ("(" ++ show ids1 ++ "," ++ typs2 ++ ")") wher])
	(_, VarT ids2 n2) -> if kinds_only then return (False, Just [])
				else if check_only then
				    return (False, Nothing)
				else do
				    typs1 <- liftIO (do_kshow limit TopShowSpan False typ1)
				    tic1' <- remap (ctx2, sp2) tic1
				    case verbose of
					Just h -> do
					    let TypeInContext t1' _ = tic1'
					    t1str' <- do_kshow limit TopShowSpan False t1'
					    liftIO $ hPutStrLn h $ "remapped to " ++ t1str'
					Nothing -> return ()
				    return $ (True, Just [Unifier tic2 (return tic1') ("(" ++ show ids2 ++ "," ++ typs1 ++ ")") wher])
	(NameT nm1 _, NameT nm2 _) -> if kinds_only then return (False, Just []) else case Data.Map.lookup nm1 ctx1 of
				    Nothing -> die ("Type '" ++ nm1 ++ "' is not in scope at " ++ show sp1 ++ ".")
				    Just (ByType, (TypeInContext tref1 _)) -> case Data.Map.lookup nm2 ctx2 of
					    Nothing -> die ("Type '" ++ nm2 ++ "' is not in scope at " ++ show sp2 ++ ".")
					    Just (ByType, (TypeInContext tref2 _)) -> do
						are_same <- tref1 `same_tref` tref2
						return (False, if are_same then Just [] else Nothing)
					    otherwise -> die (nm2 ++ " is not a type name.")
				    otherwise -> die (nm1 ++ " is not a type name.")
	(AndT _ t1s, AndT _ t2s) -> if length t1s == length t2s then do
				    let type_pairs = zip t1s t2s
				    unifieds <- mapMMaybe (\(t1, t2) -> (u2' t1 t2)) type_pairs
				    return $ mapSnd (fmap concat) unifieds
				 else return type_mismatch
	(OrT t1s, OrT t2s) -> if length t1s == length t2s then do
				    let type_pairs = zip t1s t2s
				    unifieds <- mapMMaybe (\(t1, t2) -> (u2' t1 t2)) type_pairs
				    return $ mapSnd (fmap concat) unifieds
				 else return type_mismatch
	(NotT, NotT) -> return (False, Just [])
	(FunctionT from1 to1, FunctionT from2 to2) -> do
				(prgs1, mx1) <- u2' from1 from2
				(prgs2, mx2) <- u2' to1 to2
				return $ (prgs1 || prgs2, do
				    x1 <- mx1
				    x2 <- mx2
				    return $ x1 ++ x2)
	(CofunctionT from1 to1, CofunctionT from2 to2) -> do
				(prgs1, mx1) <- u2' from1 from2
				(prgs2, mx2) <- u2' to1 to2
				return $ (prgs1 || prgs2, do
				    x1 <- mx1
				    x2 <- mx2
				    return $ x1 ++ x2)
	(LambdaT k1 nm1 t1, LambdaT k2 nm2 t2) -> do
		    have_match <- kinds_match k1 k2
		    ensure_type_variable k1
		    ensure_type_variable k2
		    link_two_variables "4" verbose k1 k2
		    let satisfy = do
			UType _ sp1 (LambdaT k1 nm1 t1) <- liftIO $ readIORef typ1
			UType _ sp2 (LambdaT k2 nm2 t2) <- liftIO $ readIORef typ2
			have_match <- kinds_match k1 k2
			return (if have_match
			    then Just ((TypeInContext t1 (addToTypeContext ctx1 nm1 (ByType, k1))),
				       (TypeInContext t2 (addToTypeContext ctx2 nm2 (ByType, k2))),
				       \_ -> \t1 -> \t2 -> return $ t1 ++ " with " ++ t2)
			    else Nothing)
		    mu <- satisfy
		    (prgs1, mx1) <- case mu of
			Nothing -> return (False, Just [Unsatisfied satisfy
					          "internal error: Kinds could not be matched."])
			Just (tic1, tic2, where_str) -> u2'' tic1 tic2
		    (prgs2, mx2) <- u2' k1 k2
		    return $ (prgs1 || prgs2, do
			x1 <- mx1
			x2 <- mx2
			return $ x1 ++ x2)

	-- As of sym35.hs I got rid of all the other unifications that are not
	-- simply structural. Don't want to eliminate fix/Y, though.
	(YNT n, CofunctionT from to) -> y_unify n from to typ1
	(CofunctionT from to, YNT n) -> y_unify n from to typ2
	(YNT n1, YNT n2) -> u2' n1 n2
	(YNT n, AndT _ [to_p, ti_p]) | lfn -> do
		let (to, ti) = (ti_p, to_p)
		mfrom <- m_application m_not m_any ti
		case mfrom of
		    Just [from] -> y_unify n from to typ1  -- 2 stage unify
		    otherwise -> do
			    tnot <- uctorT_ sp1 NotT
			    to' <- liftIO $ newIORef (UType Proposition sp1 (ApplicationT tnot typ1))
			    u2' ti to'
	(AndT _ [to_p, ti_p], YNT n) | lfn -> do
		let (to, ti) = (ti_p, to_p)
		mfrom <- m_application m_not m_any ti
		case mfrom of
		    Just [from] -> y_unify n from to typ1
		    otherwise -> do
			    tnot <- uctorT_ sp1 NotT
			    to' <- liftIO $ newIORef (UType Proposition sp1 (ApplicationT tnot typ1))
			    u2' ti to'
	(YVT n, FunctionT from to) -> y_unify n from to typ1
	(FunctionT from to, YVT n) -> y_unify n from to typ2
	(YVT n1, YVT n2) -> u2' n1 n2
	(YVT n, ApplicationT notc_ref t) | lfn -> do
		UType _ sp _ <- liftIO $ readIORef t
		mnotc <- m_not notc_ref
		mt <- m_and_2 m_any (m_application m_not m_any) t
		case mnotc >> mt of
		    Just [from, to] -> y_unify n from to typ1  -- 2 stage unify
		    otherwise -> syny sp >>= u2' t
	(ApplicationT notc_ref t, YVT n) | lfn -> do
		UType _ sp _ <- liftIO$ readIORef t
		mnotc <- m_not notc_ref
		mt <- m_and_2 m_any (m_application m_not m_any) t
		case mnotc >> mt of
		    Just [from, to] -> y_unify n from to typ2  -- 2 stage unify
		    otherwise -> syny sp >>= u2' t

	-- Simplify type applications.
	(ApplicationT ltref arg, _)
	    -> do
		maybe_typ1' <- reduce_type_application typ1 ctx1
		case maybe_typ1' of
		    Just body' -> u2'' (TypeInContext body' ctx1) tic2
		    Nothing -> case t2 of
			ApplicationT c2 a2 -> do
			    (prgs1, mx1) <- u2' ltref c2
			    (prgs2, mx2) <- u2' arg a2
			    return $ (prgs1 || prgs2, do
				    x1 <- mx1
				    x2 <- mx2
				    return $ x1 ++ x2)
			otherwise -> return type_mismatch
	(_, ApplicationT ltref arg)
	    -> do
		maybe_typ2' <- reduce_type_application typ2 ctx2
		case maybe_typ2' of
		    Just body' -> u2'' tic1 (TypeInContext body' ctx2)
		    Nothing -> case t1 of
			ApplicationT c1 a1 -> do
			    (prgs1, mx1) <- u2' c1 ltref
			    (prgs2, mx2) <- u2' a1 arg
			    return $ (prgs1 || prgs2, do
				    x1 <- mx1
				    x2 <- mx2
				    return $ x1 ++ x2)
			otherwise -> return type_mismatch
	(ExistsT k1 nm1 t1, ExistsT k2 nm2 t2) -> do
		    have_match <- kinds_match k1 k2
		    ensure_type_variable k1
		    ensure_type_variable k2
		    link_two_variables "3" verbose k1 k2
		    let satisfy = do
			UType _ sp1 (ExistsT k1 nm1 t1) <- liftIO $ readIORef typ1
			UType _ sp2 (ExistsT k2 nm2 t2) <- liftIO $ readIORef typ2
			have_match <- kinds_match k1 k2
			return (if have_match
			    then Just ((TypeInContext t1 (addToTypeContext ctx1 nm1 (ByType, k1))),
				       (TypeInContext t2 (addToTypeContext ctx2 nm2 (ByType, k2))),
				       \_ -> \t1 -> \t2 -> return $ t1 ++ " with " ++ t2)
			    else Nothing)
		    mu <- satisfy
		    (prgs1, mx1) <- case mu of
			Nothing -> return (False, Just [Unsatisfied satisfy
						  "internal error: Kinds could not be matched."])
			Just (tic1, tic2, where_str) -> u2'' tic1 tic2
		    (prgs2, mx2) <- u2' k1 k2
		    return $ (prgs1 || prgs2, do
			x1 <- mx1
			x2 <- mx2
			return $ x1 ++ x2)
	(ForallT k1 nm1 t1, ForallT k2 nm2 t2) -> do
		    have_match <- kinds_match k1 k2
		    ensure_type_variable k1
		    ensure_type_variable k2
		    link_two_variables "4" verbose k1 k2
		    let satisfy = do
			UType _ sp1 (ForallT k1 nm1 t1) <- liftIO $ readIORef typ1
			UType _ sp2 (ForallT k2 nm2 t2) <- liftIO $ readIORef typ2
			have_match <- kinds_match k1 k2
			return (if have_match
			    then Just ((TypeInContext t1 (addToTypeContext ctx1 nm1 (ByType, k1))),
				       (TypeInContext t2 (addToTypeContext ctx2 nm2 (ByType, k2))),
				       \_ -> \t1 -> \t2 -> return $ t1 ++ " with " ++ t2)
			    else Nothing)
		    mu <- satisfy
		    (prgs1, mx1) <- case mu of
			Nothing -> return (False, Just [Unsatisfied satisfy
						    "internal error: Kinds could not be matched."])
			Just (tic1, tic2, where_str) -> u2'' tic1 tic2
		    (prgs2, mx2) <- u2' k1 k2
		    return $ (prgs1 || prgs2, do
			x1 <- mx1
			x2 <- mx2
			return $ x1 ++ x2)
	otherwise -> return $ type_mismatch
     where
{-
	y_unify :: TypeInContext -> TypeInContext -> TypeInContext -> TypeInContext -> StateT [TypeId] IO (Maybe [Unifier])
	y_unify n from to other = do
				mx1 <- u2' n to
				mx2 <- u2' other from
				return $ (prgs1 || prgs2, do
					x1 <- mx1
					x2 <- mx2
					return $ x1 ++ x2)
-}
	y_unify :: UTRef -> UTRef -> UTRef -> UTRef -> StateT [TypeId] IO (Bool, Maybe [Unifier])
	y_unify n from to other = do
				(prgs1, mx1) <- u2' n to
				(prgs2, mx2) <- u2' other from
				return $ (prgs1 || prgs2, do
					x1 <- mx1
					x2 <- mx2
					return $ x1 ++ x2)
	u2' t1 t2 = u2'' (TypeInContext t1 ctx1) (TypeInContext t2 ctx2)
	u2'' = u2 limit
	syny :: Span -> StateT [TypeId] IO UTRef
	syny sp = do
	    rt1 <- unoT sp
	    not_t <- uctorT_ sp NotT
	    rt2 <- liftIO $ newIORef (uprop sp $ ApplicationT not_t rt1)
	    rt3 <- unoT sp
	    liftIO $ newIORef (uprop sp $ AndT FunctionOrigin [rt3, rt2])
	do_show tp = liftIO (do_kshow limit TopShowSpan True tp)

reduce_type_application :: UTRef -> TypeContext -> StateT [TypeId] IO (Maybe UTRef)
reduce_type_application typ ctx = do
    UType k1 sp1 t <- liftIO $ readIORef typ
    case t of
	ApplicationT ltref arg -> reduce_type_application' ltref arg ctx
	otherwise -> return Nothing
reduce_type_application' :: UTRef -> UTRef -> TypeContext
				-> StateT [TypeId] IO (Maybe UTRef)
reduce_type_application' ltref arg ctx = do
	maybe_ltref' <- reduce_type_application ltref ctx
	let ltref' = case maybe_ltref' of
		Just typ' -> typ'
		Nothing -> ltref
	lt <- liftIO $ readIORef ltref'
	case lt of
	    UType lk sp (LambdaT k nm body) -> fmap Just $ substitute_param nm arg body
	    UType lk sp (NameT nm _) -> case Data.Map.lookup nm ctx of
		Nothing -> die ("Type '" ++ nm ++ "' is not in scope at "
						++ show sp ++ ".")
		Just (ByType, (TypeInContext ntref nctx))
			-> reduce_type_application' ntref arg nctx
		otherwise -> die (nm ++ " is not a type name.")
	    otherwise -> return Nothing
    where
	substitute_param :: VarName -> UTRef -> UTRef -> StateT [TypeId] IO UTRef
	substitute_param name subst_tref tref = threadUTRefTransform f tref
	    where
		f tref = do
		    tp <- liftIO $ readIORef tref
		    case tp of 
			UType k sp (NameT name' _) | name' == name -> return subst_tref
			otherwise -> return tref

ensure_type_variable :: UTRef -> StateT [TypeId] IO ()
ensure_type_variable tref = do
	UType k sp tp <- liftIO $ readIORef tref
	case tp of
		VarT _ _ -> return ()
		otherwise -> do
			wrong_tp <- do_kshow default_type_limit ShowSpan False tp
			die $ "need a type variable, have: " ++ wrong_tp

mapSnd f (a,b) = (a,f b)

-- Sequence as long as Nothing is not returned.
-- The Bool represents "progress", independent of failures.
mapMMaybe :: (Monad m) => (a -> m (Bool, Maybe b)) -> [a] -> m (Bool, Maybe [b])
mapMMaybe f xs = sequence_maybe (map f xs)

sequence_maybe :: (Monad m) => [m (Bool, Maybe t)] -> m (Bool, Maybe [t])
sequence_maybe [] = return (False, Just [])
sequence_maybe as = sm as False id
    where
	sm [] prgs as' = return (prgs, Just (as' []))
	sm (a:as) prgs as' = do
		(prgs', mval) <- a
		let prgs'' = prgs || prgs'
		case mval of
		    Nothing -> return (prgs'', Nothing)
		    Just val -> sm as prgs'' (as'.(val:))

-- The context maps type names from the scopes affecting
-- tp to the scopes affecting the location where the remapped
-- type will reside.
-- If the context is significant then we can't copy any
-- variables from t2 into n1, or t1 into n2.
-- But if NameT's are represented as absolute locations on the stack,
-- then VarT's need not be cloned, because valid NameT's will have
-- a stable representation.  Just watch out because the unifier may
-- include a name within the type that's being re-referenced.
-- Note that the use of any stack-based numbers is problematic if full
-- unification is performed, because the same type may be defined
-- a different stack depths depending on the path to it.

{-
Tentative solution 1 is to make a pre-pass and give all type
identifiers unique names. But I don't like that because it seems to be
overkill.  Global uniqueness shouldn't be necessary when the language
will require references to names to obey scope rules.  Moreover,
type checking should verify the scope rule, and the use of
global unique names is to solve some technical difficulty in tracking
scopes in a way which gives up on the ability to track scopes well
enough to simultaneously veryfiy the scope rule.  I think both can
be done at the same time.

Tentative solution 2 is to not do any sharing of type objects in the
expression/type tree.  When a unifier copies a type, NameT's within
the type are resolved to somewhere in the stack, and the copy of
the NameT is checked to ensure that the corresponding name definition is part
of its stack. NameT's are given type node references when resolved,
and the context stack of type nodes is carried around.

Beyond that, tentative solution 2 must make sure that matching
continues wherever appropriate when a variable has been 

The context stack has been envisioned with two roles. Are they
compatible? 
 1. resolution of NameT's
 2. translation of NameT's when cloning a branch of the type tree
They appear compatible because both purposes rely on a stack of
names that are based on scope.  However the translation used in
the earlier version of type inference made an assumption that
the items on the stack match one-for-one with items on the other
stack.  This is not valid beyond the part of the stack which
deals with quantifiers.

Because abstract and generic type names are scoped based on
expressions, representing them in the context requires some extra
work.  

-}

-- Everything in the remapped tp is made into an independent copy.
-- However, VarT's that are duplicated remain in the same clique.
-- It's just that substitution for such a VarT will require remapping
-- any NameT's in the new context.  Thus, unification of such a VarT
-- requires its new context, which is not a problem.

remap :: (TypeContext, Span) -> TypeInContext -> StateT [TypeId] IO TypeInContext
remap (ctx2_0, sp2_0) tic0@(TypeInContext tpref0 ctx1) = remap' 0 ctx2_0 tic0
  where
    remap' :: Integer -> TypeContext -> TypeInContext -> StateT [TypeId] IO TypeInContext
    remap' n ctx2 tic@(TypeInContext tpref ctx1) = do
	    tp_str <- do_kshow default_type_limit TopShowSpan False tpref
	    -- liftIO $ hPutStrLn h $ "remapping " ++ show n ++ " " ++ tp_str ++ "\nfrom ctx1 of " ++ show (keys ctx1) ++ "\nto ctx2 of " ++ show (keys ctx2)
	    UType k sp tp <- liftIO $ readIORef tpref
	    let finish_without_check (tp :: Tp UTRef) = do
		    k' <- fmapM r' k
		    tref <- ((liftIO $ newIORef (UType k' sp tp)) :: StateT [TypeId] IO UTRef)
		    case k of
			VarKind id kref -> do
			    -- Link it in.
			    kvar_refs <- fmap (kref :) (more_kvars kref kref)
			    if not (tref `elem` kvar_refs) then
				    Kind.relink (last kvar_refs) tref
				else return ()
			otherwise -> return ()
		    return $ TypeInContext tref ctx2
	    let finish (tp :: Tp UTRef) = do
		    case tp of
			VarT _ _ -> die "remap losing a type ref"
			otherwise -> return ()
		    finish_without_check tp
	    let add_type_name ctor k1 nm t = do
		    flat_k1 <- liftIO $ extract k1
		    k2 <- prep flat_k1
		    tic' <- r (addToTypeContext ctx2 nm (ByType, k2))
			      (TypeInContext t (addToTypeContext ctx1 nm (ByType, k1)))
		    finish $ ctor k2 nm (drop_context tic')
	    case tp of
		VarT ids ref -> do
			    -- Get the id that the new var links with.
			    tvar_refs <- fmap (ref :) (more_tvars ("7" ++ " for " ++ show ids) ref ref)
			    id <- get_id ref
			    -- Create the new VarT object.
			    new_tic@(TypeInContext new_ref _) <- finish_without_check (VarT id ref)
			    -- Link it in.
			    relink (last tvar_refs) new_ref
			    cc "13" ref
			    cc "14" new_ref
			    return new_tic
		NameT nm orig -> do  -- !!! Inefficient and buggy.
			    case Data.Map.lookup nm ctx1 of
				Just (ByType, TypeInContext tref _) -> do
				    maybe_remapped_name <- findTypeInContext ctx2 tref
				    case maybe_remapped_name of
					    Just remapped_name -> finish $ NameT remapped_name orig
					    otherwise -> do
						tpref_str <- do_kshow default_type_limit TopShowSpan False tpref0
						die ("The type " ++ tpref_str ++ " is used at " ++ show sp2_0 ++ ", where the name " ++ nm ++ " is hidden or not in scope.")
				Just _ -> die ("The name '" ++ nm ++ "' is not a type name at " ++ show sp ++ ".")
				otherwise -> die ("The name '" ++ nm ++ "' is not in scope at " ++ show sp ++ ".")
		NoT -> finish tp
		AndT origin ts -> do
			    ts' <- mapM r' ts
			    finish $ AndT origin ts'
		OrT ts -> do
			    ts' <- mapM r' ts
			    finish $ OrT ts'
		NotT -> finish NotT
		FunctionT fromt tot -> do
			    fromt' <- r' fromt
			    tot' <- r' tot
			    finish $ FunctionT fromt' tot'
		CofunctionT fromt tot -> do
			    fromt' <- r' fromt
			    tot' <- r' tot
			    finish $ CofunctionT fromt' tot'
		ApplicationT constr arg -> do
			    constr' <- r' constr
			    arg' <- r' arg
			    finish $ ApplicationT constr' arg'
		LambdaT k nm t -> add_type_name LambdaT k nm t
		YVT t -> do
			    t' <- r' t
			    finish $ YVT t'
		YNT t -> do
			    t' <- r' t
			    finish $ YNT t'
		ExistsT k nm t -> add_type_name ExistsT k nm t
		ForallT k nm t -> add_type_name ForallT k nm t
		UnsatisfiedT matchmaker msg -> finish $ tp
	where
	    r = remap' (n+1)
	    r' t = fmap drop_context $ r ctx2 (TypeInContext t ctx1)
	    drop_context = drop_type_context

drop_type_context (TypeInContext tref _) = tref

double_check_unified :: UseFunctionTypes -> Maybe Handle -> Bool
		      -> [(TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)]
		      -> Integer
		      -> Map VarName (Way, TypeInContext)
                      -> StateT [TypeId] IO (Maybe String)
double_check_unified ufn verbose verbose_kinds matches limit builtins = case matches of
	(match:further_matches) -> do
		check_result <- double_check_match ufn verbose verbose_kinds match
							  limit builtins
		case check_result of
			Just message -> return (Just message)
			Nothing -> double_check_unified ufn verbose verbose_kinds further_matches limit builtins
	[] -> return Nothing

double_check_match :: UseFunctionTypes -> Maybe Handle -> Bool
		      -> (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String) -> Integer
		      -> Map VarName (Way, TypeInContext)
		      -> StateT [TypeId] IO (Maybe String)
double_check_match ufn verbose verbose_kinds (tic1@(TypeInContext tref1 ctx1), tic2@(TypeInContext tref2 ctx2), wher) limit builtins = do
	msg <- mkmsg limit verbose_kinds wher tref1 tref2
	aside verbose ("double_check_match\nat " ++ msg
	    {- ++ "\nwith ctx1 of " ++ show (keys ctx1) ++ "\nwith ctx2 of " ++ show (keys ctx2) -})
	(progress, mus) <- unify2 ufn verbose verbose_kinds False True True limit msg tic1 tic2
	case mus of
	    Nothing -> do
		t1@(UType k1 _ _) <- liftIO (readIORef tref1)
		t2@(UType k2 _ _) <- liftIO (readIORef tref2)
		msg <- mkmsg limit (k1 /= k2) wher tref1 tref2
		return $ Just ("Internal error, found during double-check: " ++ msg)
	    Just (u:us) -> return $ Just $ "Internal error: Unify2 found a unifier during double-check.\nThe unifier is " ++ (
		case u of
		    Unifier _ _ msg' wher -> "Unifier: " ++ msg' ++ ", at " ++ msg
		    KindUnifier _ _ msg' -> "KindUnifier: " ++ msg'
		    Unsatisfied _ msg' -> "Unsatisfied: " ++ msg')
	    Just [] -> return (if progress
				then Just "Internal error: Unify2 applied a unifier during double-check."
				else Nothing)

type_names = [c:as| as<-listsOf ['a'..'z'], c<-['A'..'Z']]
get_type_name names = head (type_names \\ names)

instance FoldableTree (Tp Type) where
    -- fold_tp :: (Eq t) => (Tp -> [t] -> t) -> Tp -> t
    -- fold_tp f t = f t (foldr union [] sublist)
    foldTree f t = f t (map ftf $ typeChildrenOf t)
	where
	    ftf (Type k sp tp) = foldTree f tp

typeChildrenOf :: Tp typ -> [typ]
typeChildrenOf tp = case tp of
		VarT ids ref -> []
		NameT nm _ -> []
		AndT _ ts -> ts
		OrT ts -> ts
		NotT -> []
		NoT -> []
		FunctionT t1 t2 -> [t1, t2]
		CofunctionT t1 t2 -> [t1, t2]
		ApplicationT c a -> [c, a]
		LambdaT k nm t -> [t]
		YVT t -> [t]
		YNT t -> [t]
		ExistsT k nm t -> [t]
		ForallT k nm t -> [t]
		UnsatisfiedT matchmaker msg -> []

instance Positionable Type where
	set_pos sp (Type k _ t) = Type k sp t
	get_span (Type _ sp _) = sp
