/*
  harmonograph is Copyright (c) Prescott K. Turner, 2006. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
*/
#include "math.h"
#include "pendulums.h"
#include "wx/wx.h"
#include "wx/txtstrm.h"
#include <ctype.h>

using namespace std;

const double pi = atan(1.0)*4.0;

const double g = 386.0886; // inches per second per second

int round2i(double d) { return d >= 0 ? int(d+0.5) : int(d-0.5); }

double length_to_freq(double length) {
    return sqrt(g/length)/(2*pi);
}

// Pendulums
Pendulums::Pendulums(double x_freq, double y_freq,
		     double x_phase, double y_phase,
		     double x_energy, double y_energy,
                     double friction_magnitude
		):
	x_pendulum(x_freq, x_phase, x_energy),
	y_pendulum(y_freq, y_phase, y_energy),
	time(0.0),
	friction_magnitude(friction_magnitude),
	paused(false),
	done(false)
{
}

bool Pendulums::advance(double time_increment) {
    if (!done && !paused) {
	pendulums_history.push_front(get_state());

	// Apportion friction between the two pendulums.
	double x_position, x_momentum;
	x_pendulum.get_state(x_position, x_momentum);
	double y_position, y_momentum;
	y_pendulum.get_state(y_position, y_momentum);
	double momentum = sqrt(x_momentum*x_momentum+y_momentum*y_momentum);
	double position = sqrt(x_position*x_position+y_position*y_position);
	double x_friction_force = friction_magnitude
			    * (momentum > 0.0 ? -x_momentum/momentum : 0.0);
	double y_friction_force = friction_magnitude
			    * (momentum > 0.0 ? -y_momentum/momentum : 0.0);
	bool x_adv = x_pendulum.advance(time_increment, x_friction_force, friction_magnitude);
	bool y_adv = y_pendulum.advance(time_increment, y_friction_force, friction_magnitude);
	done = !(x_adv || y_adv);
    }
    time += time_increment;
    return (!done && !paused);
}

void Pendulums::clear_history() {
     pendulums_history.clear();
     }

PendulumsState Pendulums::get_state() {
    PendulumsState ps;
    x_pendulum.get_state(ps.x.position, ps.x.momentum);
    y_pendulum.get_state(ps.y.position, ps.y.momentum);
    ps.time = time;
    return ps;
}

list<PendulumsState> &Pendulums::get_history() {
    return pendulums_history;
}

void Pendulums::pause(bool p) {
    paused = p;
}

Pendulum::Pendulum(double freq, double phase, double energy)
  : frequency(freq)
{
    set_energy(energy, phase);
}

void Pendulum::set_state(double position, double momentum) {
    state.position = position;
    state.momentum = momentum;
}

// Sets pendulum energy, with the given phase.
void Pendulum::set_energy(double energy, double phase) {
    set_state(energy*cos(phase), -energy*sin(phase));
}

void Pendulum::get_state(double &position, double &momentum) {
    position = state.position;
    momentum = state.momentum;
}

// Add the second value to the first, but if the first value would
// cross 0.0, stop at 0.0.
double diminish(double signed_value, double friction_impulse) {
    return signed_value > 0.0
		? (signed_value + friction_impulse > 0.0 ?
			signed_value + friction_impulse : 0.0)
	        : (signed_value + friction_impulse < 0.0 ?
			signed_value + friction_impulse : 0.0);
}

bool Pendulum::advance(double time, double friction_force, double friction_magnitude) {
    // ds/dt = 2*pi*frequency*p
    // dp/dt = -2*pi*frequency*s
    double delta_phase = 2*pi*frequency*time;
    double cosp = cos(delta_phase);
    double sinp = sin(delta_phase);
    PendulumState original = state;
    if (original.momentum == 0.0
            && fabs(original.position*sinp)<friction_magnitude*time) {
	// The restorative force does not overcome [simplified] friction.
	return false;
    }
    else {
        double momentum1 = original.momentum*cosp - original.position*sinp;
        state.momentum = diminish(momentum1, friction_force*time);
        state.position = original.position*cosp + original.momentum*sinp;
	return true;
    }
}

wxOutputStream& Pendulums::save_history(wxOutputStream& stream) {
    wxTextOutputStream text_stream( stream );
    
    text_stream << "begin evolution" << '\n';

    list<PendulumsState>::reverse_iterator iter;
    list<PendulumsState> &history = pendulums_history;
    PendulumsState *ps0 = 0;
    for (iter = history.rbegin(); iter != history.rend(); iter++) {
	PendulumsState &ps1 = *iter;
	if (ps0 != NULL) {
	    text_stream << (ps1.time - ps0->time) << ' ' << g << '\n';
	}
        text_stream << ps1.x.position << ' ' << ps1.x.momentum << ' ' << ps1.y.position << ' ' << ps1.y.momentum << '\n';
	ps0 = &ps1;
    }
    if (!done) {
        PendulumsState ps1 = get_state();
        if (ps0 != NULL) {
            text_stream << (ps1.time - ps0->time) << ' ' << g << '\n';
        }
        text_stream << ps1.x.position << ' ' << ps1.x.momentum << ' ' << ps1.y.position << ' ' << ps1.y.momentum << '\n';
    }
    text_stream << "end evolution" << '\n';
    
    return stream;
}

void pass_space(wxInputStream& stream) {
    while(isspace(stream.Peek())) stream.GetC();
}

wxInputStream& Pendulums::load_history(wxInputStream& stream) {
    pendulums_history.clear();
    wxTextInputStream text_stream( stream );
    
    text_stream.ReadLine(); // begin evolution
    time = 0.0;
    if (stream.Peek() != 'e') {
	double position, momentum;
	position = text_stream.ReadDouble();
	momentum = text_stream.ReadDouble();
	x_pendulum.set_state(position, momentum);
	position = text_stream.ReadDouble();
	momentum = text_stream.ReadDouble();
	y_pendulum.set_state(position, momentum);
	while ((pass_space(stream), stream.Peek()) != 'e' && !stream.Eof()) {
	    pendulums_history.push_front(get_state());
	    time += text_stream.ReadDouble();
	    /* g = */ text_stream.ReadDouble();

	    position = text_stream.ReadDouble();
	    momentum = text_stream.ReadDouble();
	    x_pendulum.set_state(position, momentum);
	    position = text_stream.ReadDouble();
	    momentum = text_stream.ReadDouble();
	    y_pendulum.set_state(position, momentum);
	}
    }
    text_stream.ReadLine();

    return stream;
}
