/*
  harmonograph is Copyright (c) Prescott K. Turner, 2006. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
*/
#include "wx/wx.h"
#include "wx/txtstrm.h"

#include "trialdoc.h"
#include "trialview.h"

using namespace std;

IMPLEMENT_DYNAMIC_CLASS(TrialDocument, wxDocument)

TrialDocument::TrialDocument(void)
  : x_length("36"),
    y_length("36"),
    x_phase("0"),
    y_phase("-0.25"),
    x_amplitude("0.92"),
    y_amplitude("0.92"),
    friction("0.05"),
    pendulums(NULL)
{
}

TrialDocument::~TrialDocument(void)
{
}

wxOutputStream& TrialDocument::SaveObject(wxOutputStream& stream)
{
    wxTextOutputStream text_stream( stream );
    
    text_stream << "begin parameters" << '\n';
    text_stream << "x_length " << x_length << '\n';
    text_stream << "y_length " << y_length << '\n';
    text_stream << "x_phase " << x_phase << '\n';
    text_stream << "y_phase " << y_phase << '\n';
    text_stream << "x_amplitude " << x_amplitude << '\n';
    text_stream << "y_amplitude " << y_amplitude << '\n';
    text_stream << "friction " << friction << '\n';
    text_stream << "end parameters" << '\n';
    
    if (pendulums != NULL) {
        pendulums->save_history(stream);
    }
    
    return stream;
}

bool to_double(const wxString &str, double &val, const wxString &where, wxString &diagnostic) {
    wxString opd2 = str.AfterFirst('/');
    if (opd2.IsEmpty()) {
	if (str.ToDouble(&val)) {
	    return true;
	}
	else if (str.IsEmpty()) {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", no value is provided.";
	    return false;
	}
	else {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", " << str << " is not a number.";
	    return false;
	}
    }
    else {
	wxString opd1 = str.BeforeFirst('/');
	double num1, num2;
        if (!opd1.ToDouble(&num1)) {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", " << (opd1.IsEmpty() ? str : opd1) << " is not a number.";
	    return false;
	}
        else if (!opd2.ToDouble(&num2)) {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", " << opd2 << " is not a number.";
	    return false;
	}
	else if (num2 == 0) {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", division by zero.";
	    return false;
        }
	else {
	    val = num1/num2;
	    return true;
        }
    }
}

bool phase_to_phase(const wxString &str, double &val, const wxString &where, wxString &diagnostic) {
    double v1;
    if (to_double(str, v1, where, diagnostic)) {
	val = v1*2.0*pi;
	return true;
    }
    else {
	return false;
    }
}

bool lengthstr_to_freq(const wxString &str, double &val, const wxString &where, wxString &diagnostic) {
    double v1;
    if (!to_double(str, v1, where, diagnostic)) {
	return false;
    }
    else if (v1 <= 0.0) {
	diagnostic.clear();
	diagnostic << "In " << where << ", the length must be positive.";
	return false;
    }
    else {
        val = length_to_freq(v1);
	return true;
    }
}

bool to_smallish_quantity(const wxString &str, double &val, const wxString &where, wxString &diagnostic) {
    double v1;
    if (to_double(str, v1, where, diagnostic)) {
	if (0 <= v1 && v1 <= 2) {
	    val = v1;
	    return true;
        }
	else {
	    diagnostic.clear();
	    diagnostic << "In " << where << ", the value must be between 0 and 2.";
	    return false;
	}
    }
    else {
	return false;
    }
}

// Set the pendulums with the parameters in this document.
// Return whether successful, and a diagnostic if no success.
bool TrialDocument::set_pendulums(wxString &diagnostic) {
    double y_phase_d, x_phase_d, x_freq_d, y_freq_d;
    double x_amplitude_d, y_amplitude_d, friction_d;
    bool success = phase_to_phase(y_phase, y_phase_d, "vert phase", diagnostic)
	&& phase_to_phase(x_phase, x_phase_d, "hor phase", diagnostic)
	&& lengthstr_to_freq(x_length, x_freq_d, "hor length", diagnostic)
	&& lengthstr_to_freq(y_length, y_freq_d, "vert length", diagnostic)
	&& to_smallish_quantity(x_amplitude, x_amplitude_d, "hor amplitude", diagnostic)
	&& to_smallish_quantity(y_amplitude, y_amplitude_d, "vert amplitude", diagnostic)
	&& to_smallish_quantity(friction, friction_d, "friction", diagnostic);
    if (success) {
	delete pendulums;
        pendulums = new Pendulums(x_freq_d, y_freq_d, x_phase_d, y_phase_d,
			          x_amplitude_d, y_amplitude_d, friction_d);
    }
    return success;
}

wxInputStream& TrialDocument::LoadObject(wxInputStream& stream)
{
    wxTextInputStream text_stream( stream );
    
    text_stream.ReadLine();
    text_stream.ReadWord(); x_length = text_stream.ReadWord();
    text_stream.ReadWord(); y_length = text_stream.ReadWord();
    text_stream.ReadWord(); x_phase = text_stream.ReadWord();
    text_stream.ReadWord(); y_phase = text_stream.ReadWord();
    text_stream.ReadWord(); x_amplitude = text_stream.ReadWord();
    text_stream.ReadWord(); y_amplitude = text_stream.ReadWord();
    text_stream.ReadWord(); friction = text_stream.ReadWord();
    text_stream.ReadLine();

    if (stream.Peek() == 'b') {
	wxString diagnostic;
	if (set_pendulums(diagnostic)) {
	    pendulums->load_history(stream);
	}
	else {
	    delete pendulums;
	    pendulums = NULL;
	}
    }
    else {
	delete pendulums;
	pendulums = NULL;
    }

    UpdateAllViews();

    return stream;
}

list<PendulumsState> &TrialDocument::get_pendulums_history() {
    if (pendulums != 0) {
        return pendulums->get_history();
    }
    else {
	static list<PendulumsState> *p = new list<PendulumsState>();
	return *p;
    }
}

PendulumsState TrialDocument::get_pendulums_state() {
    return pendulums->get_state();
}

void TrialDocument::cancel_motion() {
    delete pendulums;
    pendulums = NULL;
    UpdateAllViews();
}
